package org.lsst.ccs.subsystem.teststand;

import java.util.ArrayList;
import org.lsst.ccs.subsystem.teststand.limits.AlgorithmFactory;
import org.lsst.ccs.subsystem.teststand.limits.LimitAlgorithm;
import java.util.List;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystem.teststand.limits.ChannelLimits;
import org.lsst.ccs.subsystem.teststand.limits.Limits;
import org.lsst.ccs.subsystem.teststand.limits.LimitsInterface;
import org.lsst.ccs.subsystem.teststand.limits.TransitionAlgorithm;

/**
 * A class which encapsulates the thermal configuration. Typically a different
 * configuration will be loaded when the thermal target changes.
 *
 * @author tonyj
 */
public class ThermalConfiguration {

    /**
     * Cold set point
     */
    @ConfigurationParameter(category = "thermal")
    double coldSetPoint;

    /**
     * Cold set point limits
     */
    @ConfigurationParameter(category = "thermal")
    List<Double> coldSetPointLimits;

    /**
     * Cryo set point
     */
    @ConfigurationParameter(category = "thermal")
    double cryoSetPoint;

    /**
     * Cryo set point limits
     */
    @ConfigurationParameter(category = "thermal")
    List<Double> cryoSetPointLimits;

    /**
     * Algorithm to be used to compute limits, typically based on set point.
     */
    @ConfigurationParameter(category = "thermal")
    String coldMonitoringLimitAlgorithm;
    @ConfigurationParameter(category = "thermal")
    String cryoMonitoringLimitAlgorithm;

    /**
     * Parameters needed for limit algorithms.
     */
    @ConfigurationParameter(category = "thermal")
    List<Double> coldMonitoringLimitAlgorithmParameters;
    @ConfigurationParameter(category = "thermal")
    List<Double> cryoMonitoringLimitAlgorithmParameters;

    /**
     * Algorithm to be used to compute transitions.
     */
    @ConfigurationParameter(category = "thermal")
    String coldMonitoringTransitionAlgorithm;
    @ConfigurationParameter(category = "thermal")
    String cryoMonitoringTransitionAlgorithm;

    /**
     * Parameters needed for transition algorithms.
     */
    @ConfigurationParameter(category = "thermal")
    List<Double> coldMonitoringTransitionAlgorithmParameters = new ArrayList<>();
    @ConfigurationParameter(category = "thermal")
    List<Double> cryoMonitoringTransitionAlgorithmParameters = new ArrayList<>();

    LimitAlgorithm createColdLimitsAlgorithm(double nominalInitialValue, Limits initialLimits) {
        LimitAlgorithm la = AlgorithmFactory.createLimitAlgorithm(coldMonitoringLimitAlgorithm);
        Limits setPointLimits = new Limits(coldSetPointLimits);
        la.init(coldMonitoringLimitAlgorithmParameters, nominalInitialValue, initialLimits, coldSetPoint, setPointLimits);
        return la;
    }

    LimitAlgorithm createCryoLimitsAlgorithm(double nominalInitialValue, Limits initialLimits) {
        LimitAlgorithm la = AlgorithmFactory.createLimitAlgorithm(cryoMonitoringLimitAlgorithm);
        Limits setPointLimits = new Limits(cryoSetPointLimits);
        la.init(cryoMonitoringLimitAlgorithmParameters, nominalInitialValue, initialLimits, cryoSetPoint, setPointLimits);
        return la;
    }

    TransitionAlgorithm createColdTransitionAlgorithm(LimitAlgorithm limitsAlgorithm, double initialValue, ChannelLimits limits, long startTime) {
        if (coldMonitoringTransitionAlgorithm != null && !coldMonitoringTransitionAlgorithm.isEmpty()) {
            TransitionAlgorithm ta = AlgorithmFactory.createTransitionAlgorithm(coldMonitoringTransitionAlgorithm);
            ta.init(coldMonitoringTransitionAlgorithmParameters, limitsAlgorithm, initialValue, limits, startTime);
            return ta;
        } else {
            return null;
        }
    }

    TransitionAlgorithm createCryoTransitionAlgorithm(LimitAlgorithm limitsAlgorithm, double initialValue, ChannelLimits limits, long startTime) {
        if (cryoMonitoringTransitionAlgorithm != null && !cryoMonitoringTransitionAlgorithm.isEmpty()) {
            TransitionAlgorithm ta = AlgorithmFactory.createTransitionAlgorithm(cryoMonitoringTransitionAlgorithm);
            ta.init(cryoMonitoringTransitionAlgorithmParameters, limitsAlgorithm, initialValue, limits, startTime);
            return ta;
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        return "ThermalConfiguration{" + "coldSetPoint=" + coldSetPoint + ", cryoSetPoint=" + cryoSetPoint + ", coldMonitoringLimitAlgorithm=" + coldMonitoringLimitAlgorithm + ", cryoMonitoringLimitAlgorithm=" + cryoMonitoringLimitAlgorithm + ", coldMonitoringLimitAlgorithmParameters=" + coldMonitoringLimitAlgorithmParameters + ", cryoMonitoringLimitAlgorithmParameters=" + cryoMonitoringLimitAlgorithmParameters + ", coldMonitoringTransitionAlgorithm=" + coldMonitoringTransitionAlgorithm + ", cryoMonitoringTransitionAlgorithm=" + cryoMonitoringTransitionAlgorithm + ", coldMonitoringTransitionAlgorithmParameters=" + coldMonitoringTransitionAlgorithmParameters + ", cryoMonitoringTransitionAlgorithmParameters=" + cryoMonitoringTransitionAlgorithmParameters + '}';
    }


}
