package org.lsst.ccs.subsystem.teststand;

import java.util.Map;
import java.util.HashMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.xed.XED;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 ***************************************************************************
 **
 * XED device class for the teststand subsystem
 * 
 * * @author: Homer Neal
 * **************************************************************************
 */
public class XEDDevice extends Device implements Fe55Device {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("EXTENDRETRACT", Channel.TYPE_SWITCH);
    }

    private XED exretrct = null;

    private boolean isconnected = false;
    
    private int extend_retract = -1 ; // is the XED extended or retracted

    TSConfig cfg = new TSConfig();

    private String outname = "";

    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    boolean failedToInitialize = false;

    /**
     ***************************************************************************
     **
     ** XEDDevice constructor - put XED in initial subsystem state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public XEDDevice(String host, int port)
            throws DriverException {
        isconnected = true;
        try {
            exretrct = new XED();

            if (exretrct != null) {
                exretrct.openftdi(host, port); // open connection
            }
        } catch (DriverException f) {
            System.out.println("Failed to open connection to XED device!");
            isconnected = false;
        }
    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public XEDDevice() {
        try {
            exretrct = new XED();
        } catch (DriverException f) {
            System.out.println("Failed to create to XED device instance!");
        }

    }

    /**
     ***************************************************************************
     **
     ** Performs configuration. *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "XED module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (exretrct != null) {
                exretrct.close();
            }
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("XED device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (isconnected) {
            log.debug("XED device is already initialized and prompt returns:");
            try {
                log.debug(exretrct.getIdent());
            } catch (Exception e) {
                if (!failedToInitialize) {
                    log.error("XED device failed to respond to ident request! The initialization has FAILED.");
                    failedToInitialize = true;
                }
                isconnected = false;
            }
            log.debug("Setting device online.");
            setOnline(true);
            kstate = TSState.pwrstates.OK;
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected LED device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** Opens FTDI connection to a device. *
     * **************************************************************************
     */
    @Command(name = "openftdi", description = "Open FTDI connection to device")
    public String openftdi(@Argument(name = "serialname", description = "Serial device name") String host,
            @Argument(name = "baud", description = "baud rate") int port
    ) throws DriverException {
        exretrct.openftdi(host, port);
        kstate = TSState.pwrstates.OK;
        return exretrct.getIdent();
    }

  
    /**
     ***************************************************************************
     **
     ** opens the shutter *
     * **************************************************************************
     */
    @Command(name = "extendFe55", description = "extend the arm")
    public void extendFe55() {
        try {
            if (exretrct != null && isconnected) {
                exretrct.extendFe55();
                extend_retract = 1;
            }
        } catch (DriverException f) {
            log.error("Failed open/close the XED");
        }
    }

    /**
     ***************************************************************************
     **
     ** closes the shutter *
     * **************************************************************************
     */
    @Command(name = "retractFe55", description = "retract the arm")
    public void retractFe55() {
        try {
            if (exretrct != null && isconnected) {
                exretrct.retractFe55();
                extend_retract = 0;
            }
        } catch (DriverException f) {
            log.error("Failed open/close the XED");
        }
    }

    /**
     ***************************************************************************
     **
     ** returns the position of the arm
     * **************************************************************************
     */
    @Command(name = "getExtend_retract", description = "get the position of the arm")
    public int getExtend_retract() {
        return extend_retract;
    }

    public void setExtend_retract(int extend_retract) {
        this.extend_retract = extend_retract;
    }


    /**
     ***************************************************************************
     **
     ** Converts a boolean to on/off. *
     * **************************************************************************
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_SWITCH) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_SWITCH) {
                log.debug("setting channel online - chan=" + chan);
                setOnline(true);
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    @Command(name = "reset", description = "reset trips etc...")
    public void reset() throws DriverException {
        if (exretrct != null) {
            exretrct.reset();
        }
        kstate = TSState.pwrstates.OK;

    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("XEDDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if (type == Channel.TYPE_SWITCH) {
                value = extend_retract;
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    @Command(name = "setstate", description = "set XED device status")
    public void setState(int istate) {
        kstate = TSState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get XED device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
