package org.lsst.ccs.subsystem.teststand;

//import java.util.HashSet;
//import java.util.Set;
import java.util.Map;
import java.util.HashMap;

//import org.lsst.ccs.bus.messages.AidaPlots;
import hep.aida.util.XMLUtils;
import hep.aida.*;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.lsst.ccs.bus.data.KeyValueDataList;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.keithley.N2502;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;
import org.lsst.ccs.subsystem.teststand.data.PDState;

/**
 ***************************************************************************
 **
 ** Device class for communicating with the Keithley driver
 *
 * @author Homer Neal *
 * **************************************************************************
 */
public class Keithley2502Device extends Device implements BiasDevice, PhotoDiodeDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("TEMP", Channel.TYPE_TEMP);
        typeMap.put("VOLTS", Channel.TYPE_VOLTS);
        typeMap.put("CURR", Channel.TYPE_POWER);
        typeMap.put("PRESSURE", Channel.TYPE_UNKNOWN);
        typeMap.put("WAVELENGTH", Channel.TYPE_SWITCH);
        typeMap.put("PA", Channel.TYPE_UNKNOWN);
    }

    private String host;
    private int port;
    private final static int EVENT_ID_BIAS = 0;

//    protected Scpi scpi = new Scpi();
    private N2502 n25;

    private boolean isconnected = false; // is the device connected and ready to communicate
    private double nplc = 0; // period in terms of # of pulse line counts for buffered reads
    private int nreads = 1; // #reads for buffered reads

    TSConfig cfg = null;

    private double[] runBias = new double[TSConfig.MAXSTATES]; // value overriden by groovy configuration

    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    boolean failedToInitialize = false;

//    IAnalysisFactory af = null;
    IDataPointSetFactory dpsf = null;
    private double last_curr;
    private double last_volt;
    private int plotidx = 0;
    private boolean doPDPlot = false;

    /**
     ***************************************************************************
     **
     ** KeithleyDevice constructor - put Keithley in initial subsystem state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running *
     * **************************************************************************
     */
    public Keithley2502Device(String host, int port)
            throws DriverException {
        this.host = host;
        this.port = port;
        isconnected = true;
        n25 = new N2502();
        try {
            if (n25 != null) {
                n25.openftdi(host, port); // open connection
            }
        } catch (DriverException f) {
            System.out.println("Failed to open connection to Keithley device!");
            isconnected = false;
        }
        if (isconnected) {

            /*
             if (n25 != null) {
             n25.setVoltage(0.0); // set voltage to 0.
             n25.setOutput(TSState.onOff.OFF == TSState.onOff.ON);
             }
             */
            if (n25.getOutput()) {
                kstate = TSState.pwrstates.ON;
            } else {
                kstate = TSState.pwrstates.OFF;
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public Keithley2502Device() {
//        scpi = n25;
    }

    public void setCfg(TSConfig cfg) {
        this.cfg = cfg;
    }

    /**
     ***************************************************************************
     **
     ** Performs configuration. *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "Keithley module";
    }

    /**
     ***************************************************************************
     **
     ** Opens connection to a device using all defaults *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device using defaults")
    public void open() throws DriverException {
        if (n25 != null) {
            n25.open();
        }
        kstate = TSState.pwrstates.OK;
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (n25 != null) {
                n25.close();
            }
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("Keithley device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    @Command(name = "init", description = "(re) initialize connection")
    protected void initialize() {
        n25.setOKTOTALK(true);
        if (isconnected) {
            log.debug("Keithley device is already initialized and connected to:");
            try {
                log.debug(n25.printdevid());
            } catch (Exception e) {
                try {
                    log.info("Trying to reconnect to Keithley device at address " + host);
                    n25.close();
                    n25.openftdi(host, port); // open connection
                    log.debug(n25.printdevid());
                } catch (DriverException ex) {
                    if (!failedToInitialize) {
                        log.error("Keithley device failed to respond to ident request! The initialization has FAILED.");
                        failedToInitialize = true;
                    }
                    isconnected = false;
                }
            }
            log.debug("Setting device online.");
            setOnline(true);
            kstate = TSState.pwrstates.OK;
        } else if (!failedToInitialize) {
            log.error("Tried to initialize unconnected bias device!");
            failedToInitialize = true;
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the data point set factory address *
     * **************************************************************************
     */
    public void setDPSF(IDataPointSetFactory dpsf) {
        if (isconnected) {
            this.dpsf = dpsf;
        }
    }

    /**
     ***************************************************************************
     **
     ** Opens connection to a device. *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device")
    public String open(@Argument(name = "host", description = "Host name") String host
    ) throws DriverException {
        n25.open(host);
        kstate = TSState.pwrstates.OK;
        return n25.printdevid();
    }

    /**
     ***************************************************************************
     **
     ** Opens connection to a device. *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device")
    public String open(@Argument(name = "host", description = "Host name") String host,
            @Argument(name = "port", description = "Port number") int port
    ) throws DriverException {
        n25.open(host, port);
        kstate = TSState.pwrstates.OK;
        return n25.printdevid();
    }

    /**
     ***************************************************************************
     **
     ** Opens FTDI connection to a device. *
     * **************************************************************************
     */
    @Command(name = "openftdi", description = "Open FTDI connection to device")
    public String openftdi(@Argument(name = "serialname", description = "Serial device name") String host,
            @Argument(name = "baud", description = "baud rate") int port
    ) throws DriverException {
        n25.openftdi(host, port);
        kstate = TSState.pwrstates.OK;
        return n25.printdevid();
    }

    /**
     ***************************************************************************
     **
     ** Sets ARM count *
     * **************************************************************************
     */
    @Command(name = "setarmcount", description = "Set the ARM count")
    public void setArmCount(@Argument(name = "count", description = "ARM count") int count) {
        if (isconnected) {
            try {
                n25.setArmCount(count);
            } catch (DriverException e) {
                log.error("Failed to set Arm Count:" + e);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets TRIGger count *
     * **************************************************************************
     */
    @Command(name = "settrigcount", description = "Set the trigger count")
    public void setTrigCount(@Argument(name = "value", description = "trigger count") int value) throws DriverException {
        if (isconnected) {
            n25.setTrigCount(value);
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the Display ON(true)/OFF(false) *
     *
     * @param dstate
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    @Command(name = "setdisplay", description = "Set the display")
    public void setDisplay(@Argument(name = "OnOff", description = "true(ON)/false(OFF)") boolean dstate) {
        if (isconnected) {
            try {
                n25.setDisplay(dstate);
            } catch (DriverException e) {
                log.error("Failed to set display to state " + dstate + " : " + e);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Clear buffer *
     * **************************************************************************
     */
    @Command(name = "clrbuff", description = "Clears the Keithley read buffer")
    public void clearBuffer() throws DriverException {
        if (isconnected) {
            n25.clrBuff();
        }
    }

    /**
     ***************************************************************************
     **
     ** accumBuffer: start the accumulation of readings into the devices
     * internal buffer
     *
     * **************************************************************************
     * @param nreads
     * @param nplc
     */
    @Command(type = CommandType.QUERY, name = "accumBuffer", description = "start the buffer accumulation")
    public double accumBuffer(
            @Argument(name = "nreads", description = "#reads (armcount)") int nreads,
            @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc) {
        accumBuffer(nreads, nplc, true);

        log.info("publish that accumBuffer is done");
//        publish("accumBuffer","done");
        PDState pdState = new PDState(PDState.ACCUM_DONE, (int) System.currentTimeMillis());
//        getSubsystem().publishStatus(PDState.KEY, pdState);

        return (nplc / 60.0 * nreads);
    }

    /**
     ***************************************************************************
     **
     ** accumBuffer: start the accumulation of readings into the devices
     * internal buffer
     *
     * **************************************************************************
     * @param nreads
     * @param nplc
     * @param wait
     * @return
     */
    @Command(type = CommandType.QUERY, name = "accumBuffer", description = "start the buffer accumulation")
    public double accumBuffer(
            @Argument(name = "nreads", description = "#reads (armcount)") int nreads,
            @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc,
            @Argument(name = "wait", description = "to wait (true) or not to (false)") boolean wait) {
        if (isconnected) {
            try {
                this.nreads = nreads;
                this.nplc = nplc;
                cfg.setPDcnt(nreads);
                cfg.setPDnplc(nplc);
                n25.accumBuffer(nreads, nplc, wait);
            } catch (DriverException e) {
                log.error("Failed to prep device for doing accumulations." + e);
            }
        }
        return (nplc / 60.0 * nreads);
    }

    /**
     * Wait for accumulation to complete
     * 
     * @param timeout
     */
    @Command(type = CommandType.QUERY, description = "Wait for data accumulation to complete")
    public void waitAccum(@Argument(description = "Timeout (secs)") double timeout) {
    }

    /**
     ***************************************************************************
     **
     ** Read buffer *
     * **************************************************************************
     * @return
     */
    @Command(type = CommandType.QUERY, name = "readBuffer", description = "read the buffer")
    public double[][] readBuffer() {
        double[][] buff = null;
        try {
            buff = n25.readBuffer();
        } catch (DriverException e) {
            log.error("Failed to read buffer!" + e);
        }
        try {
            n25.clrBuff(); // clear the buffer
//            n25.reset(); // reset the reading count
        } catch (DriverException e) {
            log.error("Failed to reset the device after buffer read!" + e);
        }

        if (isDoPDPlot()) {

            log.error("doing new aidaplots");
//        AidaPlots plots = new AidaPlots();
            KeyValueDataList plots = new KeyValueDataList();
            log.debug("declaring pdthst");
            plotidx = (plotidx % 3) + 1;
            IDataPointSet pdthst = dpsf.create("CurrentHist" + plotidx, "Current Reading vs. time(s) - Date:" + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                    System.currentTimeMillis())), 2);

            log.debug("filling pdthst");

            // Fill the two dimensional IDataPointSet
            for (int i = 0; i < this.nreads && i < buff[0].length; i++) {
                pdthst.addPoint();
//            pdthst.point(i).coordinate(0).setValue((double) i * nplc);
                pdthst.point(i).coordinate(0).setValue(buff[1][i]);
                pdthst.point(i).coordinate(0).setErrorPlus(0.);
                pdthst.point(i).coordinate(1).setValue(buff[0][i]);
                pdthst.point(i).coordinate(1).setErrorPlus(0.);
                pdthst.point(i).coordinate(1).setErrorMinus(0.);
            }

            try {
                log.debug("doing new addplots");

                //Fill the AidaPlot object with their XML representation
                plots.addData("dataPointSet", XMLUtils.createXMLString((IManagedObject) pdthst));
            } catch (IOException ex) {
                log.error("Unable to create XML rep of AidPlot object!", ex);
            }

//Publish the plots somewhere in your subsystem
            log.info("publishing plot");
//        publish("PDtimeHist", plots);
            KeyValueDataList kvd = new KeyValueDataList();
            kvd.addData("PDtimeHist", plots);
            s.publishSubsystemDataOnStatusBus(kvd);
        }

//        publish("photodiodebuff",buff);
        return buff;
    }

    /**
     ***************************************************************************
     **
     ** Read buffer and dump it to a file
     *
     *
     * @param filename
     * **************************************************************************
     * @return
     */
    @Command(type = CommandType.QUERY, name = "readBuffer", description = "read the buffer and dump to file")
    public double[][] readBuffer(@Argument(name = "filename", description = "output filename for the Keithley buffered values") String filename) {
        double[][] buff = readBuffer();
        FileWriter fstream = null;
        File pdFl = new File(filename);
//        if (pdFl.exists()) {
        try {
            if (pdFl.exists()) {
                pdFl.delete();
            }
            pdFl.createNewFile();
        } catch (Exception e) {
            log.error("Unable to create file (" + filename + ") for reason " + e);
        }
        try {
            fstream = new FileWriter(pdFl);
        } catch (IOException e) {
            log.error("Failed to open writer stream for file (" + filename + ") for reason " + e);
        }
        try {
            if (fstream != null) {
                log.error("writing file of buffered values from either Bias or PhotoDiode device");
//            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(fstream));
                BufferedWriter out = new BufferedWriter(fstream);

                String line = null;
                for (int i = 0; i < buff[0].length; i++) {
                    line = buff[1][i] + " " + buff[0][i];
                    try {
                        out.write(line);
                        out.newLine();
                        out.flush();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }

                }
                try {
                    out.close();
                } catch (IOException e) {
                    log.error("Failed to close file!");
                }

            }
        } catch (Exception ee) {
            log.error("Failed to remove and create a new file.!");

        }
        return buff;
    }

    @Command(name = "readBufferStr", description = "read the buffer and returns as a string")
    public String readBufferStr() throws DriverException {
        double data[][] = n25.readBuffer();
        return "Data = " + getString(data[0]) + "\n"
                + " Times = " + getString(data[1]) + "\n";
    }

    /**
     ***************************************************************************
     **
     ** Set buffer size *
     * **************************************************************************
     * @param value
     */
    @Command(name = "setBuffSize", description = "Set the buffer size.")
    public void setBuffSize(@Argument(name = "value", description = "buffer size") int value) {
        if (isconnected) {
            try {
                n25.setBuffSize(value);
            } catch (DriverException ex) {
                log.error("Failed to set Keithley buffer size!" + ex);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage range *
     * **************************************************************************
     * @param value
     */
    @Command(name = "setVoltageRange", description = "Set the voltage range. Can be 10V, 50V or 500V")
    public void setVoltageRange(@Argument(name = "value", description = "Voltage Range to set") double value) {
        try {
            if (isconnected) {
                n25.setVoltageRange(value);
            }
        } catch (DriverException e) {
            log.error("Failed to set voltage range on Bias" + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage. *
     * **************************************************************************
     * @param value
     */
    @Command(name = "setVoltage", description = "Set the voltage")
    public void setVoltage(@Argument(name = "value", description = "Voltage to set") double value) {
        try {
            if (kstate.equals(TSState.pwrstates.OK) || kstate.equals(TSState.pwrstates.OFF)) {
                if (isconnected) {
                    n25.setVoltage(value);
                }
            } else {
                log.error("Unable to set output because state = " + kstate.toString());
            }
        } catch (DriverException f) {
            log.error("Failed to set the Keithley voltage!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the channel. *
     * **************************************************************************
     * @param value
     */
    @Command(name = "setChannel", description = "Select the channel")
    public void setChannel(@Argument(name = "value", description = "Channel to select") int value) {
        if (isconnected) {
            n25.setChannel(value);
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage to the value needed for acquisition. *
     * **************************************************************************
     * @param cfgstate
     */
    @Command(name = "setVoltageAcq", description = "Set the voltage to the acquisition set point")
    public void setVoltageAcq(int cfgstate) {
        double value = getRunBias(cfgstate);
        rampVolts(5.0, value);
    }

    /**
     ***************************************************************************
     **
     ** Ramps to the desired voltage *
     * **************************************************************************
     * @param duration
     * @param value
     */
    @Command(type = CommandType.QUERY, name = "rampVolts", description = "ramp the voltage")
    public void rampVolts(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Voltage to ramp to") double value) {
        try {
            if (isconnected) {
                n25.rampVolts(duration, value);
            }
        } catch (DriverException f) {
            log.error("Keithley driver failed to ramp volts!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Ramps to the desired voltage over a given time and for nsteps
     *
     * @param duration
     * @param value
     * @param nsteps
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "rampVolts", description = "ramp the voltage")
    public void rampVolts(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Voltage to ramp to") double value,
            @Argument(name = "nsteps", description = "number of steps") int nsteps) {
        try {
            if (isconnected) {
                n25.rampVolts(duration, value, nsteps);
            }
        } catch (DriverException f) {
            log.error("Keithley driver failed to ramp volts!");
        }
    }

    @Command(description = "return last current measurement")
    public double getLastCurrent() {
        return last_curr;
    }

    @Command(description = "return last voltage measurement")
    public double getLastVoltage() {
        return last_volt;
    }

    /**
     ***************************************************************************
     **
     ** Shows the set voltage. *
     * **************************************************************************
     * @return
     */
    @Command(name = "getVoltage", description = "Show the set voltage")
    public double getVoltage() throws DriverException {
        return n25.getVoltage();
    }

    /**
     ***************************************************************************
     **
     ** Reads the actual voltage. *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "readVoltage", description = "Read the actual voltage")
    public double readVoltage() {
        double volt = 0.0;
        try {
            if (n25.isOKTOTALK()) {
                volt = n25.readVoltage();
                last_volt = volt;
            } else {
                System.out.print("x");
                volt = last_volt; // see readCurrent remark
            }
        } catch (DriverException e) {
            log.error("Keithley driver failed to read voltage!");
        }
        return (volt);
    }

    /**
     ***************************************************************************
     **
     ** Sets the soft voltage limit. *
     * **************************************************************************
     */
    @Command(name = "setVoltageLimit", description = "Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name = "value",
            description = "Voltage limit to set") double value) throws DriverException {
        if (isconnected) {
            n25.setVoltageLimit(value);
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the soft voltage limit. *
     * **************************************************************************
     */
    @Command(name = "getVoltageLimit", description = "Show the soft voltage limit")
    public String getVoltageLimit() throws DriverException {
        return "Voltage limit = " + n25.getVoltageLimit();
    }

    /**
     ***************************************************************************
     **
     ** Sets the current range *
     * **************************************************************************
     */
    @Command(name = "setCurrentRange", description = "Set the current range")
    public void setCurrentRange(@Argument(name = "value", description = "Current Range to set") double value) {
        if (isconnected) {
            try {
                n25.setCurrentRange(value);
            } catch (DriverException ex) {
                log.error("Failed to set the current range: " + ex);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the current. *
     * **************************************************************************
     */
    @Command(name = "setCurrent", description = "Set the current")
    public void setCurrent(@Argument(name = "value", description = "Current to set") double value) {
        try {
            if (isconnected) {
                n25.setCurrent(value);
            }
        } catch (DriverException f) {
            log.error("Keithley driver failed to set current!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Zero correct the current *
     * **************************************************************************
     */
    @Command(name = "zeroCorrectCurrent", description = "Zero correct the current")
    public void zeroCorrectCurrent() throws DriverException {
        if (isconnected) {
            n25.zeroCorrectCurrent();
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the set current. *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "getCurrent", description = "Show the set current")
    public double getCurrent() {
        double curr = 0.0;
        try {
            curr = n25.getCurrent();
        } catch (DriverException e) {
            log.error("Keithley driver failed to get the current!!");
        }
        return (curr);
    }

    /**
     ***************************************************************************
     **
     ** Reads the actual current. *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "readCurrent", description = "Read the actual current")
    public double readCurrent() {
        log.debug("KeithleyDevice executing readcurrent operation:");

        double curr = 0.0;
        if (isconnected) {
            try {
                if (n25.isOKTOTALK()) {
                    curr = n25.readCurrent();
                    last_curr = curr;
                } else {
                    System.out.print("x");
                    curr = last_curr; // I want this to return a reception but there is a lot to change before I can do that
                }
            } catch (DriverException e) {
                log.error("Keithley driver failed to retrieve the current measurement!");
            }
        }

        return (curr);
    }

    /**
     ***************************************************************************
     **
     ** Tells whether PD accumulation is in progress *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "isAccumInProgress", description = "Are the current readings being accumulated")
    public boolean isAccumInProgress() {
        log.debug("checking is accumulation in progress");

        boolean inProgress = false;
        if (isconnected) {
            inProgress = !n25.isOKTOTALK();
        }

        return (inProgress);
    }

    /**
     ***************************************************************************
     **
     ** Sets the soft current limit. *
     * **************************************************************************
     */
    @Command(name = "setCurrentLimit", description = "Set the soft current limit")
    public void setCurrentLimit(@Argument(name = "value",
            description = "Current limit to set") double value) {
        if (isconnected) {
            try {
                n25.setCurrentLimit(value);
            } catch (DriverException ex) {
                log.error("Keithley driver failed to set the current limit! " + ex);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the soft current limit. *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "showcurrlim", description = "Show the soft current limit")
    public String getCurrentLimit() {
        double cur_lim = 0.0;
        try {
            cur_lim = n25.getCurrentLimit();
        } catch (DriverException ex) {
            log.error("Keithley driver failed to get the current limit! " + ex);
        }
        return "Current limit = " + cur_lim;
    }

    /**
     ***************************************************************************
     **
     ** Sets the read rate in PLC *
     * **************************************************************************
     */
    @Command(name = "setRate", description = "sets the read rate in plc")
    public void setRate(@Argument(name = "rate",
            description = "number of PLC") double rate) {
        try {
            if (isconnected) {
                n25.setRate(rate);
            }
        } catch (DriverException e) {
            log.error("Keithley driver failed to set the read rate!" + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Turns the output on or off. *
     * **************************************************************************
     */
    @Command(name = "setOutput", description = "Turn output on or off. 1 ==> ON")
    public void setOutput(@Argument(name = "state",
            description = "Output state: on or off") int state) {
        try {
            if (kstate.equals(TSState.pwrstates.OK) || kstate.equals(TSState.pwrstates.OFF)) {
                if (isconnected) {
                    n25.setOutput(state == 1);
                }
            } else {
                log.error("Unable to set output because state = " + kstate.toString());
            }
        } catch (DriverException e) {
            log.error("Keithley driver failed to set the output state!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the output state. *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "showoutput", description = "Show the output state")
    public boolean showOutput() {
        boolean state = (1 == 0);
        try {
            state = n25.getOutput();
        } catch (DriverException e) {
            log.error("Keithley driver failed to read output state!");
        }
        return (state);
    }

    /**
     ***************************************************************************
     **
     ** Converts an array of numbers to a string. *
     * **************************************************************************
     */
    public StringBuilder getString(double[] values) {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }

    /**
     ***************************************************************************
     **
     ** Converts a boolean to on/off. *
     * **************************************************************************
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     ***************************************************************************
     **
     ** Sets the timeout
     * **************************************************************************
     * @param nsec
     */
    @Command(name = "setTimeout", description = "set timeout for Keithley SCPI communication")
    public void setTimeout(@Argument(name = "nsec",
            description = "length of timeout in seconds") double nsec) {
        try {
            n25.setTimeout(nsec);
        } catch (DriverException ex) {
            log.error("Failed to set keithley scpi timeout:" + ex);
        }
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_VOLTS && iType != Channel.TYPE_POWER) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_VOLTS) {
                log.debug("setting channel online - chan=" + chan);
                setOnline(true);
            }
            if (type == Channel.TYPE_POWER) {
                log.debug("setting channel online - chan=" + chan);
                setOnline(true);
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    /**
     * reset - reset the Keithley to a known default state
     *
     */
    @Command(name = "reset", description = "reset trips etc...")
    public void reset() {
        if (!kstate.equals(TSState.pwrstates.TRIPPED)) {
            if (isconnected) {
                try {
                    n25.reset();
                    log.info("Keithley reset performed.");
                    n25.setTrip(false);
                    log.info("reset trip, if there was one");
                } catch (Exception ex) {
                    log.error("failed attemp to reset device:" + ex);
                }
            }
        } else {
            log.error("Unable to set output because state = " + kstate.toString()
                    + "MUST USE SUBSYSTEM resetTrip COMMAND TO RECOVER");
        }

        kstate = TSState.pwrstates.OFF;

    }

    /**
     * softReset - reset the Keithley to a known default state w/o initializing
     * all settings
     *
     */
    @Command(name = "softReset", description = "soft reset etc...")
    public void softReset() {
        if (!kstate.equals(TSState.pwrstates.TRIPPED)) {
            if (isconnected) {
                try {
                    n25.softReset();
                    log.info("Keithley soft reset performed.");
                    n25.setTrip(false);
                } catch (Exception ex) {
                    log.error("failed attemp to reset device:" + ex);
                }
            }
        } else {
            log.error("Unable to set output because state = " + kstate.toString()
                    + "MUST USE SUBSYSTEM resetTrip COMMAND TO RECOVER");
        }

        kstate = TSState.pwrstates.OFF;

    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     * @param chan
     * @param type
     * @return
     */
    @Override
    protected double readChannel(int chan, int type
    ) {
        double value = 0;
        log.debug("KeithleyDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if (type == Channel.TYPE_VOLTS) {
                value = readVoltage();
            }
            if (type == Channel.TYPE_POWER) {
                value = readCurrent();
            }
            if (value < -998. || value > 9.8E36) {
                value = Double.NaN;
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the bias value for acquisition *
     * **************************************************************************
     */
    @Command(name = "setrunbias", description = "Sets the bias value for acquisition")
    public void setRunBias(double runBias, int cfgstate
    ) {
        this.runBias[cfgstate] = runBias;
        return;
    }

    /**
     ***************************************************************************
     **
     ** Returns the bias value for acquisition *
     * **************************************************************************
     */
    @Command(type = CommandType.QUERY, name = "getRunBias", description = "Returns the bias value for acquisition")
    public double getRunBias(int cfgstate
    ) {
        return (runBias[cfgstate]);
    }

    @Command(name = "abort", description = "abort or clear abort for any long operation")
    public void setAbort(boolean state
    ) {
        n25.setAbort(state);
    }

    @Command(name = "setstate", description = "set keithley device status")
    public void setState(int istate
    ) {
        if (istate == TSState.pwrstates.TRIPPED.ordinal()) {
            n25.setTrip(true);
        }
        kstate = TSState.pwrstates.values()[istate];
    }

    @Command(type = CommandType.QUERY, name = "getState", description = "get keithley device status")
    public int getState() {
        return (kstate.ordinal());
    }

    public boolean isDoPDPlot() {
        return doPDPlot;
    }

    @Command(description = "set whether PD plots are produced")
    public void setDoPDPlot(@Argument(description = "true for do plots") boolean doPDPlot) {
        this.doPDPlot = doPDPlot;
    }

    @Command(name = "send", description = "write a string to the device")
    public void send(String str
    ) {
        try {
            System.out.println("Sending command: " + str);
            n25.writeKthly(str);
        } catch (DriverException e) {
            log.error(e);
        }
    }

    @Command(name = "send", description = "write a string to the device")
    public void send(String str1, String str2
    ) {
        try {
            String line = str1 + " " + str2;
            System.out.println("Sending command: " + line);
            n25.writeKthly(line);
        } catch (DriverException e) {
            log.error(e);
        }
    }

    @Command(name = "query", description = "write a command and read response")
    public String query(String str
    ) {
        String rtstr = "";
        try {
            System.out.println("Sending command: " + str);
            rtstr = n25.readStringKthly(str);
        } catch (DriverException e) {
            log.error(e);
        }
        return (rtstr);
    }

    @Command(name = "query", description = "write a command and read response")
    public String query(String str1, String str2
    ) {
        String rtstr = "";
        try {
            String line = str1 + " " + str2;
            System.out.println("Sending command: " + line);
            rtstr = n25.readStringKthly(line);
        } catch (DriverException e) {
            log.error(e);
        }
        return (rtstr);
    }
}
