package org.lsst.ccs.subsystem.teststand;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.apcpdu.APC7900;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  APC7900 device class for the teststand subsystem
 *
 *  @author: Owen Saxton
 */
public class APC7900Device extends Device {

    public final static int
        TYPE_OUTLET = 0,
        TYPE_POWER = 1,
        CHAN_CURRENT = 0,
        CHAN_POWER = 1;

    final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("OUTLET", TYPE_OUTLET);
        typeMap.put("POWER", TYPE_POWER);
    }

    String node;
    String user = "apc";
    String passwd = "apc";

    final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final APC7900 pdu = new APC7900();
    final List<String> outlets = new ArrayList<>();
    int[] outNums;
    Map<String, Integer> outNumMap = new HashMap<>();
    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "node", "is not specified");
        }
        fullName = "APC7900 PDU (" + node + ")";
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(APC7900.ConnType.TELNET, node, user, passwd);
            outNumMap = pdu.getOutletNumberMap();
            outNums = new int[outlets.size()];
            initSensors();
            setOnline(true);
            kstate = TSState.pwrstates.OK;
            sLog.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                sLog.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
        }
        outNumMap.clear();
        kstate = TSState.pwrstates.NOTCONFIGURED;
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  Two-element array containing the encoded type [0] and subtype [1] 
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        int index = 0;
        if (iType == null) {
            MonitorLogUtils.reportError(sLog, name, "channel type", type);
        }
        if (iType == TYPE_OUTLET) {
            index = outlets.size();
            outlets.add(subtype);
        }
        else {
            if (hwChan != CHAN_CURRENT && hwChan != CHAN_POWER) {
                MonitorLogUtils.reportError(sLog, name, "hardware channel", hwChan);
            }
        }
        
        return new int[]{iType | (index << 16), 0};
    }


    /**
     *  Initializes a channel.
     *
     *  @param  name     The channel name
     *  @param  id       The channel ID
     *  @param  hwChan   The hardware channel
     *  @param  type     The encoded channel type
     *  @param  subtype  The encoded channel subtype
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            if ((type & 0xffff) == TYPE_OUTLET) {
                int index = type >> 16;
                String outlet = outlets.get(index);
                Integer outNum = outNumMap.get(outlet);
                if (outNum == null) {
                    MonitorLogUtils.reportError(sLog, name, "outlet name", outlet);
                }
                outNums[index] = outNum;
            }
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        switch (type & 0xffff) {
        case TYPE_OUTLET:
            int outNum = outNums[type >> 16];
            try {
                value = pdu.isOutletOn(outNum) ? 1.0 : 0.0;
            }
            catch (DriverException e) {
                sLog.error("Error getting outlet state: " + e);
                setOnline(false);
            }
            break;

        case TYPE_POWER:
            try {
                value = hwChan == CHAN_POWER ? pdu.readPower() : pdu.readCurrent()[0];
            }
            catch (DriverException e) {
                sLog.error("Error reading power/current value: " + e);
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Disables the PDU.
     *
     *  Needed to relinquish control to the web or another telnet client
     */
    @Override
    @Command(name="disable", description="Disable the connection to the PDU")
    public void disable()
    {
        super.disable();
        setOnline(false);
    }


    /**
     *  Enables the PDU.
     */
    @Override
    @Command(name="enable", description="Enable the connection to the PDU")
    public void enable()
    {
        super.enable();
    }


    /**
     *  Gets the list of outlet names.
     *
     *  @return  The list of names
     */
    @Command(name="getOutletNames", description="Get the list of outlet names")
    public List<String> getOutletNames()
    {
        return new ArrayList<>(outNumMap.keySet());
    }


    /**
     *  Gets the map of outlet on states.
     *
     *  @return  The map of outlet names to on states (true or false)
     *  @throws  DriverException
     */
    @Command(name="getOutletOnStateMap", description="Get the map of outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        return pdu.getOutletOnStateMap();
    }


    /**
     *  Turns an outlet on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOn", description="Turn outlet on")
    public void outletOn(@Argument(name="name", description="Outlet name")
                         String name) throws DriverException
    {
        pdu.delayedOutletOn(getOutletNumber(name));
    }


    /**
     *  Turns an outlet off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOff", description="Turn outlet off")
    public void outletOff(@Argument(name="name", description="Outlet name")
                          String name) throws DriverException
    {
        pdu.delayedOutletOff(getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOn", description="Force outlet to turn on")
    public void forceOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        pdu.setOutletOn(getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOff", description="Force outlet to turn off")
    public void forceOutletOff(@Argument(name="name", description="Outlet name")
                               String name) throws DriverException
    {
        pdu.setOutletOff(getOutletNumber(name));
    }


    /**
     *  Gets whether an outlet is on.
     *
     *  @param  name  The outlet name
     *  @return  Whether outlet is turned on
     *  @throws  DriverException
     */
    @Command(name="isOutletOn", description="Get whether outlet is on")
    public boolean isOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        return pdu.isOutletOn(getOutletNumber(name));
    }


    /**
     *  Gets an outlet number from its name.
     *
     *  This routine should be unnecessary since the PDU handles names directly,
     *  but there's a bug in the PDU parser which causes it to report an error
     *  if the name is hyphenated.
     *
     *  @param  name  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    int getOutletNumber(String name) throws DriverException
    {
        Integer outNum = outNumMap.get(name);
        if (outNum == null) {
            throw new DriverException("Invalid outlet name");
        }
        return outNum;
    }

}
