package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state in which one blade set is retracting while the other is extending,
 * forming a traveling slit. Thread-safe.
 * @author tether
 */
class Traveling extends SimpleState<Prod> {
    private static final Logger LOG = Logger.getLogger(Traveling.class.getName());

    /**
     * Saves the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Traveling(final Prod context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.TRAVELING);
        getContext().getActions().setPhysicalState(PhysicalState.TRAVELING);
    }

    @Override
    public void brakePowerLoss(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        getContext().getActions().signalAReset();
    }

    /**
     * {@inheritDoc} Publishes the motion profile on the status bus.
     */
    @Override
    public void motionDone(final Channel<EventReply> chan, final MotionDonePLC profileData) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        final Prod ctx = getContext();
        ctx.makeTransition(ctx.getClosingState(),
                () -> ctx.getActions().sendProfile(profileData));
    }
}
