package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.*;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state machine enters this state when the CCS subsystem and the shutter PLC
 * are out of sync, that is, the state of one didn't correspond with the state of the other.
 * Thread-safe.
 * @author tether
 */
class SyncError extends SimpleState<TopContext> {

    private static final Logger LOG = Logger.getLogger(SyncError.class.getName());

    /**
     * Saves the reference to the context object.
     * @param context The context object.
     * @throws NullPointerException if the context is null.
     */
    public SyncError(final TopContext context) {
        super(context);
    }

    @Override
    public Logger getLogger() {return LOG;}

    /** {@inheritDoc} Raises a SYNC alert with a severity of ALARM. */
    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.SYNC_ERROR);
        getContext().getActions().terminateContact();
        getContext().getActions().raiseSyncAlert(getContext().getSyncAlertReason());
        getContext().getActions().setPhysicalState(PhysicalState.OTHER);
    }

    @Override
    public void resync(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        final Actions act = getContext().getActions();
        getContext().makeTransition(getContext().getSynchronizingState(), () -> {});
    }
    
    @Override
    public void contactLost(final Channel<EventReply> chan, final String msg) throws InterruptedException {
        logEvent(this);
        LOG.severe(msg);  // We're already in SyncError so just log the message.
        chan.write(ACCEPTED);
    }

}
