package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;

/**
 * The state in which the shutter, in normal operation mode, is fully open. Thread-safe.
 * @author tether
 */
class Opened extends SimpleState<Prod> {
    private static final Logger LOG = Logger.getLogger(Opened.class.getName());

    /**
     * Saves the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Opened(final Prod context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.OPENED);
        getContext().getActions().setPhysicalState(PhysicalState.OPENED);
    }

    /**
     * Requests that the shutter start closing (relayed to the PLC).
     */
    @Override
    public void closeShutter(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        final Prod ctx = getContext();
        final Actions act = getContext().getActions();
        if (act.brakeIsEngaged(ShutterSide.MINUSX) || act.brakeIsEngaged(ShutterSide.MINUSX)) {
            chan.write(new PromptReply("At least one brake is engaged."));
        }
        else {
            chan.write(PromptReply.ACCEPTED);
            ctx.makeTransition(
                    ctx.getClosingState(),
                    () -> ctx.getActions().relay(new CloseShutter()));
        }
    }

    @Override
    public void timer(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        final Prod ctx = getContext();
        final Actions act = getContext().getActions();
        if (act.brakeIsEngaged(ShutterSide.MINUSX) || act.brakeIsEngaged(ShutterSide.MINUSX)) {
            chan.write(new PromptReply("At least one brake is engaged."));
        }
        else {
            chan.write(PromptReply.ACCEPTED);
            ctx.makeTransition(ctx.getClosingState(), null);
        }
    }
}
