package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state in which the shutter is closed when in normal operating mode, that is, taking exposures.
 * Thread-safe.
 * @author tether
 */
class Closed extends SimpleState<Prod> {
    private static final Logger LOG = Logger.getLogger(Closed.class.getName());

    /** Saves the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Closed(final Prod context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.CLOSED);
        getContext().getActions().setPhysicalState(PhysicalState.CLOSED);
    }



    /**
     * Requests that the shutter start opening (relayed to the PLC).
     */
    @Override
    public void openShutter(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        final Actions act = getContext().getActions();
        if (act.brakeIsEngaged(ShutterSide.MINUSX) || act.brakeIsEngaged(ShutterSide.MINUSX)) {
            chan.write(new PromptReply("At least one brake is engaged."));
        }
        else {
            chan.write(ACCEPTED);
            getContext().makeTransition(
                getContext().getOpeningState(),
                () -> getContext().getActions().relay(new OpenShutter()));
        }
    }

    /**
     * Requests that an exposure be started. Rejected if the exposure time
     * argument is null or describes too short an exposure.
     * @param exposureTime How long the exposure should last.
     */
    @Override
    public void takeExposure(final Channel<EventReply> chan, final Duration exposureTime) throws InterruptedException {
        logEvent(this);
        final Actions act = getContext().getActions();
        if (exposureTime == null) {
            chan.write(new PromptReply("The exposure Duration musn't be null."));
        }
        else if (act.isBadExposureTime(exposureTime)) {
            chan.write(new PromptReply("The exposure time " + exposureTime + " is invalid."));
        }
        else if (act.brakeIsEngaged(ShutterSide.MINUSX) || act.brakeIsEngaged(ShutterSide.MINUSX)) {
            chan.write(new PromptReply("At least one brake is engaged."));
        }
        else {
            chan.write(ACCEPTED);
            getContext().makeTransition(
                    getContext().getOpeningState(),
                    () -> getContext().getActions().relay(new TakeExposure(exposureTime)));
        }
    }
}
