package org.lsst.ccs.subsystem.shutter.statemachine;

import java.time.Duration;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;

/**
 * The set of basic actions that the state machine can perform on its environment. An action
 * may be directed at the CCS subsystem framework, at components in the subsystem build tree
 * or at the shutter controller. Actions are the outputs of the state machine.
 * @author tether
 */
public interface Actions {

    /**
     * Determines whether the given axis is enabled, i.e., the motor is on.
     * @param side specifies the side, +X or -X, which in turn specifies the axis.
     * @return {@code true} if and only if the specified axis is enabled.
     */
    boolean axisIsEnabled(ShutterSide side);

    /**
     * Determines whether the given axis has its brake engaged. The brake is on if there's no power
     * for it or if the power relay is open.
     * @param side specifies the side, +X or -X, which in turn specifies the axis.
     * @return {@code true} if and only if the axis brake is engaged.
     */
    boolean brakeIsEngaged(ShutterSide side);
    
    /**
     * Determines whether the 24V power for the brakes is on or off.
     * @return True if and only if the power is on.
     */
    boolean brakePowerIsOn();

    /**
     * Cancels the sync timer if it's still running.
     * @see #startSyncTimer()
     */
    void cancelSyncTimer();

    /**
     * Disables the watchdog that raises an alert if we don't get any PLC messages for a long time.
     * @see #enableWatchdog()
     */
    void disableWatchdog();

    /**
     * Enables the watchdog that raises an alert if we don't get any PLC messages for a long time.
     * @see #disableWatchdog()
     */
    void enableWatchdog();

    /**
     * Determines whether the exposure time has an invalid value, for example, it's below
     * the minimum.
     * @param exposureTime The time to check.
     * @return true if the time is OK, else false.
     */
    boolean isBadExposureTime(Duration exposureTime);

    /**
     * Makes a link to the shutter controller, just enough to sent rest messages
     * and receive shutter status messages.
     * @return true if the operation succeeded, else false.
     */
    boolean makePartialContact();

    /**
     * Widens the link to the shutter controller, allowing the full exchange
     * of all message types.
     * @return true if the operation succeeded, else false.
     */
    boolean makeFullContact();

    /**
     * Immediately publishes the state of the brake power.
     * @param powerIsOn True if and only if power is known to be on.
     */
    void publishBrakePowerStatus(boolean powerIsOn);

    /**
     * Record when we enter or leave the {@code Enabled} state.
     * @param isEnabled True if entering, false if exiting.
     */
    void publishEnableStatus(boolean isEnabled);

    /**
     * Sets the physical shutter custom state that appears in the subsystem's state bundle.
     * @param newState The new state value.
     */
    void setPhysicalState(PhysicalState newState);

    /**
     * Sets the CCS software custom state that appears in the subsystem's state bundle.
     * @param newState The new state value.
     */
    void setSoftwareState(SoftwareState newState);

    /**
     * Raises a PLC alert with severity ALARM.
     * @param err The error message received from the PLC.
     */
    void raisePLCAlert(Error err);

    /**
     * Raises a SYNC alert with severity ALARM.
     */
    void raiseSyncAlert(String reason);
    
    /**
     * Raises a STOP_CMD alert with severity ALARM.
     */
    void raiseStopCmdAlert();

    /**
     * Determines if the shutter is ready for calibration. Both axes must be enabled,
     * have the brakes disengaged and be homed.
     * @return true if the conditions are met, else false.
     */
    boolean readyForCalibration();

    /**
     * Sends a reset() event to the PLC.
     */
    void resetPLC();

    /**
     * Relays an event to the PLC via the given message instance.
     */
    void relay(final MsgToPLC eventMsg);

    /**
     * Saves the calibration data to a filesystem on the CCS network.
     */
    void saveCalib(final CalibDone cal);

    /**
     * Sends the motion profile out on the CCS status bus.
     */
    void sendProfile(final MotionDonePLC mot);
    
    /**
     * Creates a task that feeds a reset() event to the central state machine.
     */
    void signalAReset();

    /**
     * Sends an event to the PLC that will start it moving one of the blade sets to its center position. The
     * CCS subsystem chooses which blade set to move in order to avoid collision.
     * @see #startSecondCentering()
     */
    void startFirstCentering();

    /**
     * Sends an event to the PLC that will start it moving the second blade set to be centered, which is the
     * one that wasn't moved by {@code startFirstCentering()}.
     * @see #startFirstCentering()
     */
    void startSecondCentering();

    /**
     * Determines if the shutter is ready for production data-taking. It has to be closed with both
     * axes enabled, homed and having the brake disengaged.
     * @return null if the shutter is ready, else a string detailing all the
     * reasons why it isn't.
     */
    String shutterIsReady();

    /**
     * Starts a timer which will generate a {@code syncYimeout()} when it expires.
     * @see #cancelSyncTimer()
     */
    void startSyncTimer();

    /**
     * Close any connection we may have to the shutter controller and stop decoding any messages
     * we have already received.
     */
    void terminateContact();
}
