package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.shutter.statemachine.BlackHoleChannel;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Keeps track of the number of PLC messages received and if none are received within a configurable
 * interval, sends a contactLost() event to the central state machine. 
 * <p>
 * @author tether
 */
public class Watchdog  implements HasLifecycle {
    private static final Logger LOG = Logger.getLogger(Watchdog.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private volatile StateMachine centralSm;

    @ConfigurationParameter(description="The interval between checks of the PLC  message counter.", units="s")
    private volatile Duration checkInterval;
    
    // Two tasks are run regularly with a third used to send the contactLost() event when needed.
    // The first task, the watchdog proper, does nothing but wait for and interpret event
    // messages. The second task, the timer, is used to send a checkMessageCount() event
    // to the first task at regular intervals.
    // 
    // The watchdog task has two states, ENABLED and DISABLED. The effects of messages
    // in the ENABLED state are:
    //     enable(): No-op.
    //     disable(): Change the state to DISABLED.
    //     countMessage(): Increment the message count.
    //     checkMessageCount(): If the message count is zero then send the contactLost() event.
    //         Otherwise clear the message count.
    // In the disabled state, all messages are no-ops except enable(), which clears the
    // message count and sets the state to ENABLED.
    //
    // The watchdog starts in the DISABLED state.
     private static enum WatchState {ENABLED, DISABLED;}

     // These two fields are set in the constructor.
    private final SynchronousChannel<Runnable> inChannel;
    private final Scheduler sched;

    // After construction only the watchdog thread is allowed to read or change these two fields.
    private int msgCount;
    private WatchState state;

    ////////// Lifecycle methods //////////

    public Watchdog() {
        this.msgCount = 0;
        this.inChannel = new SynchronousChannel<>();
        this.state = WatchState.DISABLED;
        this.sched = new Scheduler("PLCWatchdog", 3);
        this.checkInterval = Duration.ofSeconds(10);
    }

    @Override
    public void start() {
        startWatchdogTask();
        startTimerTask();
    }
    
    private void startWatchdogTask() {
        sched.schedule(this::watchdogBody, 0, TimeUnit.MILLISECONDS);
    }
    
    private void startTimerTask() {
        sched.scheduleWithFixedDelay(this::timerBody, 100, checkInterval.toMillis(), TimeUnit.MILLISECONDS);
    }
    
    private void startSenderTask() {
        sched.schedule(this::senderBody, 0, TimeUnit.MILLISECONDS);
    }

    @Override
    public void shutdown() {
        sched.shutdownNow();
    }

    ////////// Public interface when in operation //////////

    /**
     * If this is in the disabled state, switch to the enabled state.
     * @throws InterruptedException
     */
        public void enable() throws InterruptedException {
        inChannel.write(this::enableImpl);
    }

    /**
     * If this is in the enabled state, switch to the disabled state and set the count of messages received to zero.
     * @throws InterruptedException
     */
    public void disable() throws InterruptedException {
        inChannel.write(this::disableImpl);
    }

    /**
     * If this is in the enabled state, increment the count of messages received.
     * @throws InterruptedException
     */
    public void countMessage()  throws InterruptedException {
        inChannel.write(this::countImpl);
    }

    /**
     * If this is in the enabled state, raise an alert if no messages have yet been received and then reset
     * the count of messages received.
     * @throws InterruptedException
     */
    public void checkMessageCount() throws InterruptedException {
        inChannel.write(this::checkImpl);
    }


    ////////// Private implementation //////////
    private synchronized void enableImpl() {
        if (state == WatchState.DISABLED) {state = WatchState.ENABLED;}
    }

    private synchronized void disableImpl() {
        if (state == WatchState.ENABLED) {
            state = WatchState.DISABLED;
            msgCount = 0;
        }
    }

    private synchronized void countImpl() {
        if (state == WatchState.ENABLED) {++msgCount;}
    }

    private synchronized void checkImpl() {
        if (state == WatchState.ENABLED) {
            if (msgCount == 0) {
                startSenderTask();
            }
            else {
                msgCount = 0;
            }
            // At this point msgCount == 0.
        }
    }

    private void watchdogBody() {
        try {
            Thread.currentThread().setName("Watchdog main task");
            LOG.info("Starting the PLC message watchdog main task.");
            while (true) {inChannel.read().run();}
        }
        catch (InterruptedException exc) {
            LOG.info("Normal exit of the PLC message watchdog.");
        }
        catch (Exception exc) {
            LOG.log(Level.SEVERE, "Exception in watchdog task. Restarting.", exc);
            startWatchdogTask();
        }
    }

    private void timerBody() {
        try {
            Thread.currentThread().setName("Watchdog timer task");
            checkMessageCount();
        }
        catch (InterruptedException exc) {
            LOG.info("The PLC message watchdog timer was interrupted!");
        }
        catch (Exception exc) {
            LOG.log(Level.SEVERE, "Exception in the watchdog timer task.", exc);
        }
    }
    
    private void senderBody() {
        // Best-effort attempt to deliver.
        final Channel<EventReply> chan = new BlackHoleChannel<>();
        try {
            Thread.currentThread().setName("Watchdog contactLost()-sender task");
            centralSm.contactLost(chan, "No messages are coming in from the shutter PLC.");
        } catch (InterruptedException ex) {}
    }
}
