package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ALWAYS;
import static org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.PLCError;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * An implementation of {@code Actions} in which no hardware operations are performed.
 * All actions are logging at info level. Subsystem-level operations such as
 * alert handling or manipulation of the state bundle are implemented.
 * @author tether
 */
public class SimulatedActions implements Actions {
    private static final Logger LOG = Logger.getLogger(SimulatedActions.class.getName());

    private final static Alert SYNC_ALERT = new Alert("SYNC",
            "Simulated status of tracking of controller state.");
    private final static Alert PLC_ALERT = new Alert("PLC",
            "Notice of simulated shutter controller errors.");

    private final Controller control;
    private final StateMachine machine;
    private final Subsystem subsys;
    private final Publisher publish;
    private final Watchdog wdog;


    /**
     * Saves references to the components that actually implement the actions.
     * @param control A reference to the {@code Controller} component.
     * @param machine A reference to the {@code StateMachine} component.
     * @param subsys A reference to the subsystem object.
     * @param publish A reference to the {@code Publisher} component.
     * @param wdog A reference to the {@code Watchdog} component.
     */
    public SimulatedActions(
        final Controller control,
        final StateMachine machine,
        final Subsystem subsys,
        final Publisher publish,
        final Watchdog wdog)
    {
        this.control = control;
        this.machine = machine;
        this.subsys = subsys;
        this.publish = publish;
        this.wdog = wdog;
    }

    @Override
    public boolean axisIsEnabled(final ShutterSide side) {
        final ShutterStatus status = publish.getShutterStatus();
        return status.getAxisStatus(side).isEnabled();
    }

    @Override
    public boolean brakeIsEngaged(final ShutterSide side) {
        if (brakePowerIsOn()) {
            final ShutterStatus status = publish.getShutterStatus();
            // At present isBrakeEnaged() only reflects the state of the relays, not power.
            return status.getAxisStatus(side).isBrakeEngaged();
        }
        else {
            return true;
        }
    }
    
    @Override
    public boolean brakePowerIsOn() {
        return machine.brakePowerIsOn();
    }

    @Override
    public void cancelSyncTimer() {
        LOG.info("{ACTION cancelSyncTimer()}");
    }

    @Override
    public void disableWatchdog() {
        LOG.info("{ACTION disableWatchdog()}");
        try {
             wdog.disable();
        }
        catch (InterruptedException exc) {
            LOG.info("Watchdog disable was interrupted.");
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public void enableWatchdog() {
        LOG.info("{ACTION enableWatchdog()}");
        try {
             wdog.enable();
        }
        catch (InterruptedException exc) {
            LOG.info("Watchdog enable was interrupted.");
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public boolean isBadExposureTime(final Duration exposureTime) {
        LOG.info("{ACTION isBadExposure()}");
        return control.isBadExposureTime(exposureTime);
    }

    @Override
    public boolean makePartialContact() {
        LOG.info("{ACTION makePartialContact()}");
        control.simulateContact();
        return true;
    }

    @Override
    public boolean makeFullContact() {
        LOG.info("{ACTION makeFullContact()}");
        return true;
    }

    @Override
    public void publishBrakePowerStatus(final boolean powerIsOn) {
        publish.publishBrakePowerStatus(powerIsOn);
    }

    @Override
    public void publishEnableStatus(final boolean isEnabled) {
        publish.setEnableStatus(isEnabled);
    }

    @Override
    public void setPhysicalState(final PhysicalState newState) {
        LOG.info(String.format("{ACTION setPhysicalState(%s)}", newState));
        machine.setPhysicalState(newState);
    }

    @Override
    public void setSoftwareState(SoftwareState newState) {
        LOG.info(String.format("{ACTION setSoftwareState(%s)}", newState));
        machine.setSoftwareState(newState);
    }

    // Since the PLC automatically reports it status frequently, we use the ON_SEVERITY_CHANGE strategy for
    // PLC alerts in order to prevent a lot of them from piling up before the operator can take action.
    // The other alerts can only occur when a command is issued and so are recorded each time under the
    // assumption that the operator doesn't spam commands after an alert.

    @Override
    public void raisePLCAlert(final org.lsst.ccs.subsystem.shutter.plc.Error err) {
        LOG.fine("{ACTION raisePLCAlert()}");
        for (final PLCError e: err.getErrors()) {
            LOG.log(Level.WARNING, "PLC error {0} - {1}.", new Object[]{e.toString(), e.getDescription()});
        }
        final String errlist =
            err.getErrors().stream().map(PLCError::toString).collect(Collectors.joining(", "));
        Alerts.PLC.raise(subsys.getAgentService(AlertService.class), AlertState.ALARM, errlist, ON_SEVERITY_CHANGE);
    }

    @Override
    public void raiseSyncAlert(final String reason) {
        LOG.fine("{ACTION raiseSyncAlert()}");
        Alerts.SYNC.raise(subsys.getAgentService(AlertService.class), AlertState.ALARM, reason, ALWAYS);
    }
    
    @Override
    public void raiseStopCmdAlert() {
        LOG.fine("{ACTION raiseStopCmdAlert()}");
        Alerts.STOP_CMD.raise(subsys.getAgentService(AlertService.class), AlertState.ALARM, "", ALWAYS);
    }

    @Override
    public boolean readyForCalibration() {
        LOG.info("{ACTION readyForCalibration()}");
        return true;
    }

    @Override
    public void resetPLC() {
        LOG.info("{ACTION resetPLC()}");
        control.simulateResetPLC();
    }

    @Override
    public void relay(final MsgToPLC eventMsg) {
        LOG.log(Level.INFO, "'{'ACTION relay({0})'}'", eventMsg.getClass());
        control.simulateRelay(eventMsg);
    }

    @Override
    public void saveCalib(final CalibDone cal) {
        LOG.info("{ACTION saveCalib() - not yet implemented.}");
    }

    @Override
    public void sendProfile(final MotionDonePLC mot) {
        LOG.info("{ACTION sendProfile()}");
        dumpMotionDone(mot.getStatusBusMessage());
        control.checkHallTransitions(mot);
        publish.publishMotionDone(mot.getStatusBusMessage());
    }
    
    private final Scheduler resetSched = new Scheduler("Reset submitter", 1);

    @Override
    public void signalAReset() {
        final Runnable resetTask = () -> {
            try {
                final Channel<EventReply> replyChan = new SynchronousChannel<>();
                machine.reset(replyChan);
                final EventReply reply = replyChan.read();
            }
            catch (InterruptedException exc) {}
        };
        resetSched.schedule(resetTask, 0, TimeUnit.SECONDS);
    }

    private void dumpMotionDone(final MotionDone mot) {
        final StringBuilder bld = new StringBuilder();
        bld.append("[MotionDone begin]\n");
        // Header.
        bld.append("side " + mot.side() + "\n");
        bld.append("startPosition " + mot.startPosition() + " mm\n");
        bld.append("startTime " + mot.startTime().getUTCInstant().toEpochMilli() + " Unix epoch ms\n");
        bld.append("targetPosition " + mot.targetPosition() + " mm\n");
        bld.append("targetDuration " + mot.targetDuration().toMillis() + " ms\n");
        bld.append("endPosition " + mot.endPosition() + " mm\n");
        bld.append("actualDuration " + mot.actualDuration().toMillis() + " ms\n");
        // Hall transitions.
        bld.append("[Hall begin] time(Unix epoch ms) ID position(mm) isOn\n");
        for (final HallTransition hall: mot.hallTransitions()) {
            bld.append(hall.getTime().getUTCInstant().toEpochMilli());
            bld.append(" ");
            bld.append(hall.getSensorId());
            bld.append(" ");
            bld.append(hall.getPosition());
            bld.append(" ");
            bld.append(hall.isOn());
            bld.append("\n");
        }
        bld.append("[Hall end]\n");
        // Encoder samples.
        bld.append("[Encoder begin] time(Unix epoch ms) position(mm)\n");
        for (final EncoderSample enc: mot.encoderSamples()) {
            bld.append(enc.getTime().getUTCInstant().toEpochMilli());
            bld.append(" ");
            bld.append(enc.getPosition());
            bld.append("\n");
        }
        bld.append("[Encoder end]\n");
        bld.append("[MotionDone end]\n");
        LOG.fine(bld.toString());
    }

    @Override
    public void startFirstCentering() {
        LOG.info("{ACTION startFirstCentering()}");
        control.startFirstCentering();
    }

    @Override
    public void startSecondCentering() {
        LOG.info("{ACTION startSecondCentering(};}");
        control.startSecondCentering();
    }

    @Override
    public String shutterIsReady() {
        LOG.info("{ACTION shutterIsReady()}");
        return null;
    }

    @Override
    public void startSyncTimer() {
        LOG.info("{ACTION startSyncTimer()}");
    }

    @Override
    public void terminateContact() {
        LOG.info("{ACTION terminateContact()}");
    }

}
