package org.lsst.ccs.subsystem.shutter;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import static org.lsst.ccs.bus.states.AlertState.NOMINAL;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Generates {@code brakePowerChange()} events for the central state machine based on status messages
 * sent by a target subsystem, normally the quadbox subsystem (see org-lsst-ccs-subsystem-power).
 * We assume that brake power has been lost if the target subsystem does any of the following:<br>
 * <ulist>
 *     <li>Goes off-line.</li>
 *     <li>Sends a power status message in which the shutter brake voltage is too low.</li>
 *     <li>Raises a shutter brake power-loss alert of level higher than NOMINAL.</li>
 * </ulist>
 * <p>
 * We assume that brake power has been restored if the target subsystem:<br>
 * <ulist>
 *     <li>Sends a power status message in which the shutter brake voltage is high enough.</li>
 *     <li>Clears an outstanding shutter brake power-loss alert or reduces its level to NOMINAL.</li>
 * </ulist>
 * <p>It's not enough for the target subsystem to be on-line; we need to get a positive indication that
 * brake power is on.
 * <p>
 * The {@code brakeVoltageKey} configuration parameter contains the name of the PDU channel supplying
 * the voltage. It's used as the key for the voltage value in the {@code KeyValueDataList} published by the
 * quadbox subsystem periodically. It's also use as the ID of the alert raised to signal problems with
 * the voltage.
 * <p> The quadbox subsystem is recognized not by name but by it's having an agent property named
 * {@code PowerAgentProperties.QUAD_BOX_AGENT} which is transmitted with all of its messages.
 * <p>
 * Since the callbacks used to monitor the status bus should not block and the order of events must be
 * preserved, we add power-is-on flags to an unbounded queue which is read by a dedicated task. The task
 * undertakes to send the required events to the central state machine and waits for the state machine
 * to reply to each event before sending the next.
 * @author tether
 */
public class BrakePowerMonitor
        implements
        HasLifecycle,
        StatusMessageListener,
        AlertListener,
        AgentPresenceListener
{

    private static final Logger LOG = Logger.getLogger(BrakePowerMonitor.class.getName());

    @LookupField(strategy = TOP)
    private volatile Subsystem subsys;
    
    @LookupField(strategy = TREE)
    private volatile StateMachine machine;
    
    private volatile AgentMessagingLayer aml;
    
    @ConfigurationParameter(description = "The key of the brake voltage value in the KeyValueDataList.",
        isFinal = true, units="unitless")
    private volatile String brakeVoltageKey;
    
    @ConfigurationParameter(
        description = "Brake power voltage must be at least this to be considered 'on'.",
        isFinal = true, units = "V")
    private volatile double powerOnThreshold;
    
    private final BlockingQueue<Boolean> eventQueue;
    
    private final Scheduler eventScheduler;
    
    private volatile boolean lastBrakePowerFailed;
    
    BrakePowerMonitor() {
        brakeVoltageKey = "PDU_24VD/Shtr_Brakes_V";
        powerOnThreshold = 23.0;
        eventQueue = new LinkedBlockingQueue<>();
        eventScheduler = new Scheduler("brakePower", 1, new ThreadGroup("Brake power monitoring"));
        lastBrakePowerFailed = true;
    }
    
    ////////// Implementation of HasLifeCycle //////////
    /**
     * Starts listening for alerts from the target subsystem.
     */
    @Override
    public void init() {
        aml = subsys.getMessagingAccess();  // Listeners require that "aml" be initialized.
        subsys.getAgentService(AlertService.class).addListener(this);
        subsys.getAgentService(AlertService.class).startStatusAlertListening((AgentInfo a) -> true);
        LOG.info(() -> "Started listening for the alert " + brakeVoltageKey + " from the quadbox agent.");
    }
    
    /**
     * Starts listening for power status messages from the publisher.
     */
    @Override
    public void postStart() {

        aml.addStatusMessageListener(
            this,     // Call our onStatusMessage() method.
            msg ->    // Filtering.
                isQuadboxAgent(msg.getOriginAgentInfo())
                && msg instanceof StatusSubsystemData
                && ((StatusSubsystemData)msg).getObject() instanceof KeyValueDataList
        );
        LOG.info(() -> "Started listening for quadbox power-status messages.");

        aml.getAgentPresenceManager().addAgentPresenceListener(this);
        LOG.info(() -> "Started listening for agent-presence events, looking for quadbox disconnections.");

        // For information, show which quadbox agents are on-line.
        final List<AgentInfo> onlineAgents = aml.getAgentPresenceManager().listConnectedAgents();
        onlineAgents.stream()
                .filter( (AgentInfo agent) -> isQuadboxAgent(agent) )
                .forEach( (AgentInfo agent) -> {
                    LOG.log(
                            Level.INFO,
                            "Agent {0} has property {1} = {2}.",
                            new Object[]{
                                agent.toString(),  // Agent name and type.
                                PowerAgentProperties.QUAD_BOX_AGENT,
                                agent.getAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT)
                            }
                    );
                });

        // Must not start this until the state machine is running. postStart() is the last of the
        // startup lifecycle methods to be called.
        eventScheduler.schedule(this::schedulerLoop, 0/*delay*/, TimeUnit.SECONDS);
    }
    
    /**
     * Stops listening for power status messages.
     */
    @Override
    public void postShutdown() {
        eventScheduler.shutdownNow();
        LOG.info(() -> "Shutting down the loop that posts brake-power events.");
        aml.removeStatusMessageListener(this);
        LOG.info(() -> "Stopped listening for status bus messages from quadbox.");
        aml.getAgentPresenceManager().removeAgentPresenceListener(this);
        LOG.info(() -> "Stopped listening for agent-presence events for quadbox.");
        subsys.getAgentService(AlertService.class).removeListener(this);
        LOG.info(() -> "Stopped listening for alerts from quadbox.");
    }
    
    ////////// Implementation of StatusMessageListener //////////

    // Note that filtering ensures that the messages handled here are from the quadbox.
    @Override
    public void onStatusMessage(final StatusMessage rawMsg) {
        final List<KeyValueData> kvdList =
            ((KeyValueDataList)((StatusSubsystemData)rawMsg).getObject()).getListOfKeyValueData();
        Optional<KeyValueData> kvd =
            kvdList.stream()
                .filter(k -> k.getKey().equals(brakeVoltageKey))
                .findFirst();
        if (kvd.isPresent()) {
            final double voltage = (Double)kvd.get().getValue();
            final boolean powerHasFailed = (voltage < powerOnThreshold);
            // Take actions if power status has changed.
            if (powerHasFailed != lastBrakePowerFailed)
            {
                LOG.warning(() -> "Received this quadbox power message: " + rawMsg.toString());
                LOG.warning(() ->
                        brakeVoltageKey
                        + " value of "
                        + voltage
                        + "V is "
                        + (powerHasFailed ? "less than " : "greater than or equal to ")
                        + powerOnThreshold+ "V.");

            }
            lastBrakePowerFailed = powerHasFailed;
            sendPowerEvent(!powerHasFailed);
        }
    }
    
    ////////// Implementation of AlertListener //////////
        
    @Override
    public void onAlert(final AlertEvent event) {
        final String agentName = event.getSource();
        if (!isQuadboxAgentName(agentName)) {
            return;
        }
        switch (event.getType()) {
            case AGENT_CONNECTION:
                break;
            case AGENT_DISCONNECTION:
                break;
            case ALERT_CLEARED:
                {
                    // Many alerts may get cleared all at once.
                    final List<String> clearedIds = event.getClearedIds();
                    final boolean brakeAlertCleared = clearedIds.contains(brakeVoltageKey);
                    if (brakeAlertCleared) {
                        LOG.warning(() -> "The brake voltage alert was cleared by agent " + agentName + ".");
                        LOG.warning(() -> "However, we'll wait until the reported voltage is"
                                + " correct before assuming that the brakes are off.");
                    }
                }
                break;
            case ALERT_RAISED:
                {
                    final Alert alrt = event.getAlert();
                    if (alrt.getAlertId().equals(brakeVoltageKey)) {
                        LOG.warning(() -> "The brake voltage alert was raised by agent " + agentName + ".");
                        lastBrakePowerFailed = (event.getLevel() != NOMINAL);
                        sendPowerEvent(!lastBrakePowerFailed);
                    }
                }
                break;
        }
    }

    ////////// Implementation of AgentPresenceListener //////////
    ////////// All methods have default implementations that do nothing.

    @Override
    public void disconnected(AgentInfo... agents) {
        // Assumes that at most one quadbox subsystem exists.
        final Optional<AgentInfo> quadBoxDisc = Stream.of(agents)
                .filter(a -> isQuadboxAgent(a))
                .findFirst();
        if (quadBoxDisc.isPresent()) {
            LOG.warning(() -> "The quadbox agent " + quadBoxDisc.get().toString() + " has gone off-line.");
            lastBrakePowerFailed = true;
            sendPowerEvent(false);
        }
    }
    
    @Override
    public void connected(AgentInfo... agents) {
        // Assumes that at most one quadbox subsystem exists.
        final Optional<AgentInfo> quadBoxDisc = Stream.of(agents)
                .filter(a -> isQuadboxAgent(a))
                .findFirst();
        if (quadBoxDisc.isPresent()) {
            LOG.warning(() -> "The quadbox agent " + quadBoxDisc.get().toString() + " is back on-line.");
            LOG.warning(() -> "However, we'll wait until the reported voltage is"
                    + " correct before assuming that the brakes are off.");
        }
    }
    

    ////////// Helpers //////////
    public boolean isQuadboxAgent(final AgentInfo agent) {
        return agent.hasAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT);
    }
    
    // There doesn't seem to be a way to translate a subsystem name to an AgentInfo
    // other than to search for a matching name in the list of all on-line agents.
    public boolean isQuadboxAgentName(final String agentName) {
        final List<AgentInfo> onlineAgents = aml.getAgentPresenceManager().listConnectedAgents();
        final Optional<AgentInfo> alertingAgent = onlineAgents.stream()
                .filter( (AgentInfo agent) -> isQuadboxAgent(agent) )
                .filter( (AgentInfo agent) -> agent.getName().equals(agentName))
                .findFirst();
        return alertingAgent.isPresent();
    }
    
    ////////// Event-sending code //////////
    private void sendPowerEvent(final boolean powerIsOn) {
        LOG.finer(() -> "Queueing a power event with powerIsOn=" + powerIsOn);
        eventQueue.add(powerIsOn);
    }
    
    private void schedulerLoop() {
        LOG.info("Starting the brake power monitoring task.");
        final Channel<EventReply> replyChan = new SynchronousChannel<>();
        try {
            for (;;) {
                final boolean powerIsOn = eventQueue.take();
                try {
                    machine.brakePowerChange(replyChan, powerIsOn);
                    // Must wait for reply in order to preserve order of events.
                    final EventReply reply = replyChan.read();
                }
                catch (InterruptedException exc) {
                    throw exc;
                }
                catch (Exception exc) {
                    LOG.log(Level.SEVERE, "Exception in brake power monitoring task.", exc);
                }
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal termination of the brake power monitoring task.");
        }
    }

}
