package org.lsst.ccs.subsystems.shutter.gui;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.HallTransition;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.PLUSX;
import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.MINUSX;

import org.lsst.ccs.subsystems.shutter.sim.MotorPosition;
import org.lsst.ccs.subsystems.shutter.sim.MotorSimulator;
import org.lsst.ccs.subsystems.shutter.sim.CubicSCurve;
import org.lsst.ccs.subsystems.shutter.status.CloseShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.OpenShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.MoveToPositionStatus;
import org.lsst.ccs.subsystems.shutter.status.MovementHistoryStatus;
import org.lsst.ccs.subsystems.shutter.status.TakeExposureStatus;

/**
 * Coordinates updates to the plot pane and the shutter animation. This class is not thread safe
 * and its methods should be called from the Swing dispatch thread, via invokeLater() for example.
 * @author azemoon
 * @author tether
 */
public class ShutterDisplay extends JPanel {

    private final ShutterAssembly assembly;
    private PlotPanelNew plotPanel;
    private Map<ShutterSide, MotorHistory> motorHistories = new EnumMap<>(ShutterSide.class);
    private Map<ShutterSide, HallSensorHistory> hallSensorHistories = new EnumMap<>(ShutterSide.class);
    private Map<ShutterSide, EncoderReadOut> encoderReadOuts = new EnumMap<>(ShutterSide.class);
    private long timeOrigin = 0;

    public ShutterDisplay() {
        super(new BorderLayout());

        assembly = new ShutterAssembly();
        motorHistories.put(PLUSX, new MotorHistory());
        motorHistories.put(MINUSX, new MotorHistory());
        hallSensorHistories.put(PLUSX, new HallSensorHistory());
        hallSensorHistories.put(MINUSX, new HallSensorHistory());
        encoderReadOuts.put(PLUSX, new EncoderReadOut());
        encoderReadOuts.put(MINUSX, new EncoderReadOut());
        plotPanel = new PlotPanelNew(motorHistories, hallSensorHistories, encoderReadOuts);

        JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        split.add(plotPanel);
        split.add(assembly);
        add(split, BorderLayout.CENTER);
    }

    /**
     * Sets the blade set positions in the shutter assembly drawing, makes sure that it's visible
     * and repaints it.
     * @param plus the relative position of blade set PLUSX
     * @param minus the relative position of blade set MINUSX
     */
    public void setBladeSetPositions(final double plus, final double minus) {
        assembly.getShutter(PLUSX).setShutterPosition((float)plus);
        assembly.getShutter(MINUSX).setShutterPosition((float)minus);
        assembly.setVisible(true);
        assembly.repaint();
    }

    /**
     * Hides the shutter assembly picture. This is done at startup before we know the shutter positions.
     */
    public void hideAssembly() {
        assembly.setVisible(false);
        assembly.repaint();
    }

    /**
     * Prepares to display the results of a new command. Clear the plot panel,
     * set a new time origin and range for the new plots.
     */
    private void startNewCommand(long t0, float tSpan) {
        // Clear the plot panel by deleting the old data and redrawing.
        for (final ShutterSide side: ShutterSide.values()) {
            motorHistories.get(side).reset();
            hallSensorHistories.get(side).reset();
            encoderReadOuts.get(side).reset();
        }

        // Set the width of the time axis.
        plotPanel.setTimePeriod(tSpan);

        // Set the origin time to subtract from the times of motor position samples
        // and Hall transitions.
        timeOrigin = t0;
    }

    Map<ShutterSide, MotorHistory> getHistories() {
        return motorHistories;
    }

    /**
     * Plots the predicted motion for a MoveToPosition command.
     * @param moveStat the command acknowledgment sent by the worker subsystem.
     *
     * This motion is not part of a larger command involving multiple movements, so the start
     * time in moveStat is the time origin to use in the plot. Calls showPredictedMovement()
     * with arguments indicating that this is the first and the last movement in the command.
     */
    public void showPredictedMoveTo(MoveToPositionStatus moveStat) {
        startNewCommand(moveStat.getStartTime(), (float)moveStat.getMoveTime());
        showPredictedMovement(
            moveStat.getSide(),
            moveStat.getStartingPosition(),
            moveStat.getTargetPosition(),
            moveStat.getMoveTime(),
            0
        );
    }

    private final static double MOTOR_SAMPLING_INTERVAL = 0.050; // seconds

    /**
     * Plots the prediction for a single blade set movement.
     * @param side the side of the shutter that is moving.
     * @param startingPosition the initial relative position of the blade set.
     * @param targetPosition the final relative position of the blade set.
     * @param moveTime the duration in seconds of the move.
     * @param startTime the time in microseconds at which the actual command execution started.
     */
    public void showPredictedMovement(
        final ShutterSide side,
        final double startingPosition,
        final double targetPosition,
        final double moveTime,
        final long startTime
    )
    {
        System.out.println("ShutterDisplay: startingPosition " + startingPosition + " targetPosition  " + targetPosition + " moveTimeSeconds " + moveTime);

        // Run the simulation.
        final MotorSimulator motor = new MotorSimulator(
            startingPosition,
            MOTOR_SAMPLING_INTERVAL,
            1 + (int)Math.floor(moveTime / MOTOR_SAMPLING_INTERVAL), // no. of samples
            startTime,
            new CubicSCurve(targetPosition - startingPosition, moveTime)
        );

        motorHistories.get(side).addData(motor.getPositions());
    }

    void showPredictedExposure(TakeExposureStatus exposeStat) {
        final float moveTime = (float) exposeStat.getMoveTime();
        final float exposureTime = (float) exposeStat.getExposureTime();
        final ShutterSide firstSide = exposeStat.getFirstSide();
        final ShutterSide secondSide = firstSide.opposite();
        final float firstStartingPosition = assembly.getShutter(firstSide).getShutterPosition();
        final float secondStartingPosition = assembly.getShutter(secondSide).getShutterPosition();
        final float firstTargetPosition = 0;
        final float secondTargetPosition = 1;

        startNewCommand(exposeStat.getStartTime(), moveTime + exposureTime);
        showPredictedMovement(firstSide,
            firstStartingPosition,
            firstTargetPosition,
            moveTime,
            0
        );
        showPredictedMovement(secondSide,
            secondStartingPosition,
            secondTargetPosition,
            moveTime,
            (long)(1.0e6 * exposureTime)
        );
    }

    void showPredictedClose(CloseShutterStatus closeStat) {
        final float moveTime = (float)closeStat.getMoveTime();
        final ShutterSide firstSide = closeStat.getFirstSide();
        final ShutterSide secondSide = firstSide.opposite();
        System.out.println("****** ShutterDisplay closeShutter: moveTime " + moveTime + " *********");
        startNewCommand(closeStat.getStartTime(), 2.0f * moveTime);
        showPredictedMovement(firstSide,
            assembly.getShutter(firstSide).getShutterPosition(),
            0.0,
            moveTime,
            0);
        showPredictedMovement(secondSide,
            assembly.getShutter(secondSide).getShutterPosition(),
            1.0,
            moveTime,
            (long)(1.0e6 * moveTime));
    }

    void showPredictedOpen(OpenShutterStatus openStat) {
        final float moveTime = (float)openStat.getMoveTime();
        final ShutterSide firstSide = openStat.getfirstSide();
        final ShutterSide secondSide = firstSide.opposite();
        System.out.println("****** ShutterDisplay openShutter: moveTime " + moveTime + " *********");
        startNewCommand(openStat.getStartTime(), 2.0f * moveTime);
        showPredictedMovement(firstSide,
            assembly.getShutter(firstSide).getShutterPosition(),
            0.0,
            moveTime,
            0);
        showPredictedMovement(secondSide,
            assembly.getShutter(secondSide).getShutterPosition(),
            0.0,
            moveTime,
            (long)(1.0e6 * moveTime));
    }

    /**
     * Updates the plot pane and the shutter assembly drawing based on a movement in the actual shutter.
     * Plot data includes both blade positions and Hall transitions.
     * @param moveStat the movement history collected at the shutter
     */
    void showActualMovement(MovementHistoryStatus moveStat) {
        System.out.println("****** showActualMovement *********");
        final ShutterSide side = moveStat.getSide();
        showActualMotorPositions(side, moveStat.getHistory().getPositions());
        showActualHallTransitions(side, moveStat.getHistory().getHallTransitions());
    }

    /**
     * Adds actual blade positions to the plot and moves the corresponding blade set picture.
     * @param side the side of the shutter being moved
     * @param pos the collected blade set positions
     */
    private void showActualMotorPositions(ShutterSide side, List<BladePosition> pos) {
        System.out.println("********** showActualMotorPositions " + pos.size() + " ************");
        assembly.getShutter(side).setShutterPosition((float)pos.get(pos.size() - 1).getRelPosition());
        assembly.repaint();
        encoderReadOuts.get(side).addData(pos, timeOrigin);
        for (BladePosition p : pos) {
            System.out.println("UI : set " + side + " Time" + p.getTime() + " position " + p.getRelPosition());
        }
    }

    /**
     * Adds Hall transitions to the plot pane.
     * @param side the side of the shutter that was moved
     * @param trans the list of Hall transitions collected at the shutter
     */
    void showActualHallTransitions(ShutterSide side, List<HallTransition> trans) {
        System.out.println("************ showActualHallTransitions " + trans.size() + " ************");
        hallSensorHistories.get(side).addData(trans, timeOrigin);
        for (HallTransition h: trans) {
            System.out.println("UI: set " + side + " Time " + h.getTransitionTime() + " position " + h.getPosition());
            System.out.println("UI: set " + side + " ID " + h.getSensorId() + " State " + h.isOpen() + " retracting " + h.isReverse());
        }
    }

}
