package org.lsst.ccs.subsystems.shutter.gui;

import org.lsst.ccs.subsystems.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.PLUSX;
import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.MINUSX;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Arc2D;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import javax.swing.JComponent;

/**
 * Displays a diagram of the shutter; two movable rectangular blade sets on top of a circular aperture.
 * This class is not thread safe and its methods should be called from the Swing event dispatch thread.
 * @author tonyj
 * @author tether
 */
public class ShutterAssembly extends JComponent {

    private Map<ShutterSide, Shutter> shutters = new EnumMap<>(ShutterSide.class);

    public ShutterAssembly() {
        final int height = 400;
        final int width = 500;
        final int shutterWidth = 100;
        final Dimension dimension = new Dimension(width, height);
        setPreferredSize(dimension);
        setMinimumSize(dimension);
        shutters.put(PLUSX, new Shutter(shutterWidth, height));
        shutters.put(MINUSX, new Shutter(shutterWidth, height));

        // Until such time as position information is sent from the worker
        // subsystem set the shutter as closed with the -X blade set
        // extended and the +X set retracted.
        shutters.get(PLUSX).setShutterPosition(0.0f);
        shutters.get(MINUSX).setShutterPosition(1.0f);
    }

    Shutter getShutter(final ShutterSide side) {
        return shutters.get(side);
    }

    @Override
    protected void paintComponent(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        RenderingHints hints = new RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.addRenderingHints(hints);
        //Paint the camera.
        Shape circle = new Arc2D.Float(100, 50, 300, 300, 0, 360, Arc2D.CHORD);
        g2.draw(circle);

        // Paint the -X shutter on the left, extending toward the right.
        shutters.get(MINUSX).paint(g2);
        AffineTransform saveAt = g2.getTransform();
        // Paint the +X shutter ...
        g2.translate(500, 0);  // ... on the right ...
        g2.scale(-1,1);        // ... and extending toward the left.
        shutters.get(PLUSX).paint(g2);
        g2.setTransform(saveAt);

    }

    static class Shutter {

        private final List<Blade> blades = new ArrayList<Blade>();

        Shutter(int bladeWidth, int bladeHeight) {
            for (int i = 0; i < 3; i++) {
                blades.add(new Blade(bladeWidth, bladeHeight));
            }
        }

        /**
         * Set the shutter position
         * @param pos 0=retracted, 1=extended, 0.5 neither up nor down
         */
        void setShutterPosition(float pos) {
            pos = Math.max(0, Math.min(1, pos));
            blades.get(0).setPosition(100 * pos);
            blades.get(1).setPosition(200 * pos);
            blades.get(2).setPosition(300 * pos);
        }

        float getShutterPosition() {
            return blades.get(0).setPosition() / 100;
        }

        void paint(Graphics2D g2) {
            for (Blade blade : blades) {
                blade.paint(g2);
            }
        }
    }

    private static class Blade extends Rectangle.Float {

        Blade(int width, int height) {
            super(0, 0, width, height);
        }

        void setPosition(float x) {
            this.x = x;
        }

        float setPosition() {
            return this.x;
        }

        private void paint(Graphics2D g2) {
            g2.setColor(new Color(.5f, .5f, .5f, .5f));
            g2.fill(this);
            g2.setColor(Color.BLACK);
            g2.draw(this);
        }
    }

}

