package org.lsst.ccs.subsystems.shutter.gui;

import java.util.stream.Stream;

import data.DiscretePlotData;
import data.MetaData;
import data.MutablePlotData;
import data.PlotDataListener;
import data.SuggestedRange;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.freehep.util.Value;
import util.ListenerList;

import org.lsst.ccs.subsystems.shutter.sim.MotorPosition;


/**
 * Contains motor position, velocity and acceleration points generated from a simulation.
 * The simulation's sample times are assumed to be already
 * relative to the start of the shutter command.
 * @author tonyj
 * @author tether
 */
public class MotorHistory implements DiscretePlotData, MutablePlotData {

    private final List<MotorSpeed> history = new ArrayList<>();
    private final ListenerList<PlotDataListener> listeners = new ListenerList<>();
    private final String[] names = {"Time", "Acceleration", "Velocity", "Position"};
    public void MotorHistory() {
    }

    /**
     * Clears away any old data and cause the plot pane to be repainted.
     */
    public void reset() {
       history.clear();
       firePlotDataChanged();
    }

    /**
     * Adds new data and causes the plot pane to be repainted.
     * @param pos a stream of MotorPosition from simulated motion
     */
    public void addData(Stream<MotorPosition> pos) {
        MotorPosition[] apos = pos.toArray(MotorPosition[]::new);
        final double vscale =
            Arrays.stream(apos)
                .mapToDouble(p->Math.abs(p.getVelocity()))
                .max()
                .orElse(1.0);
        final double ascale =
            Arrays.stream(apos)
                .mapToDouble(p->Math.abs(p.getAcceleration()))
                .max()
                .orElse(1.0);
        Arrays.stream(apos)
        .forEach( p -> {
                history.add(
                    new MotorSpeed(
                        1e-6 * p.getTime(),
                        p.getAcceleration() / ascale,
                        p.getVelocity() / vscale,
                        p.getPosition()));                
            });
        firePlotDataChanged();
    }

    @Override
    public int getNPoints() {
        return history.size();
    }

    @Override
    public int getNDimensions() {
        return names.length;
    }

    @Override
    public String names(int index) {
        return names[index];
    }

    @Override
    public Class types(int index) {
        return Double.TYPE;
    }

    @Override
    public void getValue(Value value, int dim, int index) {
        MotorSpeed speed = history.get(index);
        switch (dim) {
            case 0: value.set(speed.getDate()); break;
            case 1: value.set(speed.getAcceleration()); break;
            case 2: value.set(speed.getVelocity()); break;
            case 3: value.set(speed.getPosition()); break;
        }
    }

    @Override
    public MetaData getMetaData() {
        return null;
    }

    @Override
    public String getTitle() {
        return "Motor History";
    }

    @Override
    public SuggestedRange getSuggestedRange() {
        return null;
    }

    @Override
    public void addPlotDataListener(PlotDataListener listener) {
        listeners.addListener(listener);
    }

    @Override
    public void removePlotDataListener(PlotDataListener listener) {
        listeners.removeListener(listener);
    }

    private void firePlotDataChanged() {
        if (!listeners.isEmpty()) {
            for (PlotDataListener l : listeners.getListeners()) {
                l.dataChanged();
            }
        }
    }

    @Override
    public Object lock() {
        return history;
    }

    private static class MotorSpeed {

        private final double date;
        private final double acceleration;
        private final double velocity;
        private final double position;

        public MotorSpeed(double date, double acceleration, double velocity, double position) {
            this.date = date;
            this.acceleration = acceleration;
            this.velocity = velocity;
            this.position = position;
        }

        public double getAcceleration() {
            return acceleration;
        }

        public double getDate() {
            return date;
        }

        public double getPosition() {
            return position;
        }

        public double getVelocity() {
            return velocity;
        }
    }

}
