package org.lsst.ccs.subsystems.shutter.gui;

import data.DiscretePlotData;
import data.MetaData;
import data.MutablePlotData;
import data.PlotDataListener;
import data.SuggestedRange;
import java.util.ArrayList;
import java.util.List;
import org.freehep.util.Value;
import util.ListenerList;

import org.lsst.ccs.subsystems.shutter.common.HallTransition;

/**
 * Contains Hall transition data gathered from the actual shutter.
 * @author azemoon
 * @author tether
 */
public class HallSensorHistory implements DiscretePlotData, MutablePlotData {

    private final List<HallData> history = new ArrayList<HallData>();
    private final ListenerList<PlotDataListener> listeners = new ListenerList<PlotDataListener>();
    private final String[] names = {"Time", "Position"};
    private static double ERROR = 0.02;

    public void HallSensorHistory() {
    }

    /**
     * Clears away all data and causes the plot pane to be repainted.
     */
    public void reset() {
       history.clear();
       firePlotDataChanged();
    }

    /**
     * Adds new Hall transitions and causes the plot pane to be repainted.
     * @param trans the list of Hall transitions
     * @param timeOrigin the start time for the shutter command
     */
    public void addData(List<HallTransition> trans, long timeOrigin) {
        trans.forEach(
            h -> {
                final double t = 1.0e-6 * (h.getTransitionTime() - timeOrigin);
                final HallData d = new HallData(t, h.getRelPosition(), ERROR);
                history.add(d);
            }
        );
        firePlotDataChanged();
    }

    @Override
    public int getNPoints() {
        return history.size();
    }

    @Override
    public int getNDimensions() {
        return names.length;
    }

    @Override
    public String names(int index) {
        return names[index];
    }

    @Override
    public Class types(int index) {
        return Double.TYPE;
    }

    @Override
    public void getValue(Value value, int dim, int index) {
        HallData transition = history.get(index);
        switch (dim) {
            case 0: value.set(transition.getTime()); break;
            case 1: value.set(transition.getPosition()); break;
            case 2: value.set(transition.getError()); break;
        }
    }

    @Override
    public MetaData getMetaData() {
        return null;
    }

    @Override
    public String getTitle() {
        return "Hall Sensor History";
    }

    @Override
    public SuggestedRange getSuggestedRange() {
        return null;
    }

    @Override
    public void addPlotDataListener(PlotDataListener listener) {
        listeners.addListener(listener);
    }

    @Override
    public void removePlotDataListener(PlotDataListener listener) {
        listeners.removeListener(listener);
    }

    private void firePlotDataChanged() {
        if (!listeners.isEmpty()) {
            for (PlotDataListener l : listeners.getListeners()) {
                l.dataChanged();
            }
        }
    }

    @Override
    public Object lock() {
        return history;
    }

    private static class HallData {

        private final double time;
        private final double position;
        private final double error;

        public HallData(double time, double position, double error) {
            this.time = time;
            this.position = position;
            this.error = error;
        }

        public double getTime() {
            return time;
        }

        public double getPosition() {
            return position;
        }

        public double getError() {
            return error;
        }
    }
}
