package org.lsst.ccs.subsystem.shutter.status;

import static java.util.Objects.requireNonNull;

/**
 * Represents the various states that the EL6688's PTP interface can be in. We normally operate
 * the device as PTPv2 Slave Only so the state transitions would be NO_OPERATION -> INITIALIZING -> 
 * LISTENING -> SLAVE.
 * @author tether
 */
public enum PtpDeviceState {
    
    /** The device has no connection to a network. Could be caused, for example,
        by a bad IP address or by bad cabling.
    */
    NO_OPERATION,
    
    /** Just got connected to a local network. */
    INITIALIZING,

    FAULTY,
    DISABLED,
    
    /** Listening for PTP packets from a master clock. */
    LISTENING,

    PRE_MASTER,
    MASTER,
    PASSIVE,
    UNCALIBRATED,
    
    /** Now sync'ed to a master clock. */
    SLAVE;

    private static final long serialVersionUID = 1L;

    /**
     * Converts a raw state number from the device into one of the enum values.
     * @param n the raw state number.
     * @return A non-null enum value.
     * @throws IllegalArgumentException if the argument is outside the legal range.
     */
    public static PtpDeviceState fromStateNumber(final int n) throws IllegalArgumentException {
        try {
            return PtpDeviceState.values()[n];
        }
        catch (final IndexOutOfBoundsException exc) {
            throw new IllegalArgumentException(
                String.format("%d is not in the range [0, %d]",
                n, PtpDeviceState.values().length - 1)
            );
        }
    }
    
    public static int toStateNumber(final PtpDeviceState state) {
        return requireNonNull(state, "Argument must be non-null.").ordinal();
    }

}
