package org.lsst.ccs.subsystem.shutter.status;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;
import org.lsst.ccs.bus.annotations.DoNotTrend;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Carries trajectory information from a single blade set motion. This carries enough information
 * to plot the actual motion as well as a prediction based on a given motion profile.
 * <p>
 * Immutable.
 * @author tether
 */
@SkipEncoding
@DoNotTrend
public final class MotionDone implements Serializable {
    private static final long serialVersionUID = 2L;

    private final ShutterSide _side;
    private final CCSTimeStamp _startTime;
    private final double _startPosition;
    private final Duration _targetDuration;
    private final double _targetPosition;
    private final double _endPosition;
    private final Duration _actualDuration;
    private final List<HallTransition> _hallTransitions;
    private final List<EncoderSample> _encoderSamples;
    private final PtpDeviceState _ptpState;
    private final int _leapSeconds;
    private final boolean _leapIsValid;

    private MotionDone(
        final ShutterSide side,
        final CCSTimeStamp startTime,
        final double startPosition,
        final Duration targetDuration,
        final double targetPosition,
        final double endPosition,
        final Duration actualDuration,
        final List<HallTransition> hallTransitions,
        final List<EncoderSample> encoderSamples,
        final PtpDeviceState ptpState,
        final int leapSeconds,
        final boolean leapIsValid
    )
    {
        this._side = side;
        this._startPosition = startPosition;
        this._startTime = startTime;
        this._targetPosition = targetPosition;
        this._targetDuration = targetDuration;
        this._endPosition = endPosition;
        this._actualDuration = actualDuration;
        this._encoderSamples = encoderSamples;
        this._hallTransitions = hallTransitions;
        this._ptpState = ptpState;
        this._leapSeconds = leapSeconds;
        this._leapIsValid = leapIsValid;
    }

    /**
     * @return The side of the shutter whose blade sets moved.
     */
    public ShutterSide side() {return _side;}

    /**
     * @return The encoder value at the time motion was started.
     */
    public double startPosition() {return _startPosition;}

    /**
     * @return The absolute time at which motion started.
     */
    public CCSTimeStamp startTime() {return _startTime;}

    /**
     * @return The intended final position of the blade set.
     */
    public double targetPosition() {return _targetPosition;}

    /**
     * @return The intended duration of the motion. This will either have
     * been provided as parameter of the motion or derived from other
     * parameters such as speed and distance.
     */
    public Duration targetDuration() {return _targetDuration;}

    /**
     * @return The actual final position of the blade set.
     */
    public double endPosition() {return _endPosition;}

    /**
     * @return The actual duration of the motion.
     */
    public Duration actualDuration() {return _actualDuration;}

    /**
     * @return The encoder samples that were taken during the motion.
     */
    public List<EncoderSample> encoderSamples() {return _encoderSamples;}

    /**
     * @return The Hall transitions that took place during the motion.
     */
    public List<HallTransition> hallTransitions() {return _hallTransitions;}
    
    /**
     * @return The state of the EL6688 PTP device.
     */
    public PtpDeviceState ptpState() {return _ptpState;}
    
    /**
     * @return The current number of leap seconds.
     */
    public int leapSeconds() {return _leapSeconds;}
    
    /**
     * @return Is the current number of leap seconds valid?
     */
    public boolean isLeapValid() {return _leapIsValid;}

    /**
     * Creates a string of the form 'MotionDone{field=value, ...}'.
     * @return The string.
     */
    @Override
    public String toString() {
        final StringBuilder bld = new StringBuilder("MotionDone{");
        bld.append("side=");
        bld.append(side());
        bld.append(", startTime=");
        bld.append(startTime());
        bld.append(", startPosition=");
        bld.append(startPosition());
        bld.append(", targetDuration=");
        bld.append(targetDuration());
        bld.append(", targetPosition=");
        bld.append(targetPosition());
        bld.append(", endPosition=");
        bld.append(endPosition());
        bld.append(", actualDuration=");
        bld.append(actualDuration());
        bld.append(", no. of Hall transitions=");
        bld.append(hallTransitions().size());
        bld.append(", no. of encoder samples=");
        bld.append(encoderSamples().size());
        bld.append(", PTP state=").append(_ptpState);
        bld.append(", leap seconds=").append(_leapSeconds);
        bld.append(", leap is valid=").append(_leapIsValid);
        bld.append("}");
        return bld.toString();
    }

    /**
     * Is this a dummy, just used to signal the end of a non-standard motion such as axis homing?
     * A dummy has no Hall transitions or encoder samples.
     * @return true if this is a dummy message, else false.
     */
    public boolean isDummy() {
        return hallTransitions().isEmpty() && encoderSamples().isEmpty();
    }

    /**
     * A builder for instances of {@code MotionsDone}. Allows you to specify the required
     * components one a a time before calling {@code build()}. The other methods each
     * return {@code this} in order to allow chaining. Hall transitions and encoder samples
     * may be added in any order since the complete list of each will be sorted
     * before constructing the instance of {@code MotionDone}.
     * <p>
     * Not thread-safe at all, as it's intended as a substitute for a public constructor.
     */
    public final static class Builder {

        private static final Logger LOG = Logger.getLogger(Builder.class.getName());

        private ShutterSide _side;
        private Double _startPosition;
        private CCSTimeStamp _startTime;
        private Double _targetPosition;
        private Duration _targetDuration;
        private Double _endPosition;
        private Duration _actualDuration;
        private List<EncoderSample> _encoderSamples;
        private List<HallTransition> _hallTransitions;
        private PtpDeviceState _ptpState;
        private Integer _leapSeconds;
        private Boolean _leapIsValid;
        private final boolean noThrow;

        /**
         * Use this constructor for normal purposes. Failed checks will
         * result in the throwing of exceptions.
         */
        public Builder() {this.noThrow = false;}

        /**
         * Use this constructor only when you need to construct faulty instances
         * of {@code MotionDone} for tests.
         * @param noThrow If true then failed checks will throw exceptions, otherwise
         * they will log warning messages.
         */
        public Builder(final boolean noThrow) {this.noThrow = noThrow;}

        /**
         * @see MotionDone#side()
         * @param side
         * @return {@code this}
         */
        public Builder side(final ShutterSide side) {_side = side; return this;}

        /**
         * @see MotionDone#startPosition()
         * @param pos
         * @return
         */
        public Builder startPosition(final double pos) {_startPosition = pos; return this;}

        /**
         * @see MotionDone#startTime()
         * @param time
         * @return {@code this}
         */
        public Builder startTime(final CCSTimeStamp time) {_startTime = time; return this;}

        /**
         * @see MotionDone#targetPosition()
         * @param pos
         * @return {@code this}
         */
        public Builder targetPosition(final double pos) {_targetPosition = pos; return this;}

        /**
         * @see MotionDone#endPosition()
         * @param pos
         * @return {@code this}
         */
        public Builder endPosition(final double pos) {_endPosition = pos; return this;}

        /**
         * @see MotionDone#targetDuration()
         * @param dur
         * @return {@code this}
         * @throws IllegalArgumentException if the duration is non-positive, unless
         * the no-throwing was chosen in the constructor.
         */
        public Builder targetDuration(final Duration dur) {
            if (dur.isNegative() || dur.isZero()) {
                if (noThrow) {
                    LOG.warning("Non-positive duration.");
                }
                else {
                    throw new IllegalArgumentException("Non-positive duration.");
                }
            }
            _targetDuration = dur;
            return this;}

        /**
         * @see MotionDone#actualDuration()
         * @param dur
         * @return {@code this}
         * @throws IllegalArgumentException if the duration is non-positive.
         */
        public Builder actualDuration(final Duration dur) {
            if (dur.isNegative() || dur.isZero()) {
                if (noThrow) {
                    LOG.warning("Non-positive duration.");
                }
                else {
                    throw new IllegalArgumentException("Non-positive duration.");
                }
            }
            _actualDuration = dur;
            return this;
        }

        /**
         * @see MotionDone#encoderSamples()
         * @param samps
         * @return {@code this}
         */
        public Builder encoderSamples(final List<EncoderSample> samps) {
            _encoderSamples = new ArrayList<>(samps);
            return this;
        }

        /**
         * Adds a single new encoder sample to any already present.
         * @see MotionDone#encoderSamples()
         * @param samp
         * @return {@code this}
         */
        public Builder addEncoderSample(final EncoderSample samp) {
            if (this._encoderSamples == null) {_encoderSamples = new ArrayList<>(100);}
            _encoderSamples.add(samp);
            return this;
        }

        /**
         * @see MotionDone#hallTransitions()
         * @param halls
         * @return {@code this}
         */
        public Builder hallTransitions(final List<HallTransition> halls) {
            _hallTransitions = new ArrayList<>(halls);
            return this;
        }

        /**
         * Appends a single new Hall transition to any already present.
         * @see MotionDone#hallTransitions()
         * @param hall
         * @return {@code this}
         */
        public Builder addHallTransition(final HallTransition hall) {
            if (_hallTransitions == null) {_hallTransitions = new ArrayList<>(100);}
            _hallTransitions.add(hall);
            return this;
        }
        
        /**
         * @see MotionDone#ptpState()
         * @param state
         * @return {@code this}
         */
        public Builder ptpState(final PtpDeviceState state) {
            _ptpState = state;
            return this;
        }
        
        /**
         * @see MotionDone#leapSeconds()
         * @param seconds
         * @return {@code this}
         */
        public Builder leapSeconds(final int seconds) {
            _leapSeconds = seconds;
            return this;
        }
        
        /**
         * @see MotionDone#isLeapValid() 
         * @param valid
         * @return 
         */
        public Builder leapIsValid(final boolean valid) {
            _leapIsValid = valid;
            return this;
        }

        /**
         * Sorts the lists of Hall transitions and encoder samples, performs some
         * checks and the constructs an instance of {@code MotionDone}.
         * @return A new instance of {@code MotionDone}.
         * @throws IllegalStateException if any of the required components are missing, unless
         * no-throwing was selected in the constructor.
         */
        public MotionDone build() {
            final List<String> missing = new ArrayList<>(20);
            if (_side == null) {missing.add("side");}
            if (_startPosition == null) {missing.add("startPosition");}
            if (_startTime == null) {missing.add("startTime");}
            if (_targetPosition == null) {missing.add("targetPosition");}
            if (_targetDuration == null) {missing.add("targetDuration");}
            if (_endPosition == null) {missing.add("endPosition");}
            if (_actualDuration == null) {missing.add("actualDuration");}
            if (_encoderSamples == null) {missing.add("encoderSamples");}
            if (_hallTransitions == null) {missing.add("hallTransitions");}
            if (_ptpState == null) {missing.add("ptpState");}
            if (_leapSeconds == null) {missing.add("leapSeconds");}
            if (_leapIsValid == null) {missing.add("leapIsValid");}
            if (missing.size() > 0) {
                final String msg =
                    String.format("Missing components of MotionDone: %s.",
                                  String.join(", ", missing));
                if (noThrow) {
                    LOG.warning(msg);
                }
                else {
                    throw new IllegalStateException(msg);
                }
            }
            Collections.sort(_encoderSamples);
            Collections.sort(_hallTransitions);
            return new MotionDone(
                _side,
                _startTime,
                _startPosition,
                _targetDuration,
                _targetPosition,
                _endPosition,
                _actualDuration,
                Collections.unmodifiableList(_hallTransitions),
                Collections.unmodifiableList(_encoderSamples),
                _ptpState,
                _leapSeconds,
                _leapIsValid
            );
        }
    }

}
