package org.lsst.ccs.subsystem.shutter.common;

import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.Optional;

/**
 * The axes available on the shutter, one per blade set. The actual axis number used
 * in the PLC and the axis name are instance fields.
 */
public enum Axis {
    AXIS0("+X", 0, true), AXIS1("-X", 1, false);

    private final String name;

    private final int plcAxisNum;

    private final boolean plusXside;

    Axis(final String name, final int plcAxisNum, final boolean plusXside) {
        this.name = name;
        this.plcAxisNum = plcAxisNum;
        this.plusXside = plusXside;
    }

    /**
     * The name used for the axis in motorplatform commands.
     * @return The axis name string.
     */
    public String getName() {
        return name;
    }

    /**
     * The axis number used by the PLC code.
     * @return The axis number.
     */
    public int getPLCAxisNum() {return plcAxisNum;}

    /**
     * Find the Axis value having the given axis name using a case-insensitive search.
     * @param name The axis name to find.
     * @return The corresponding Axis value, or null if no match was found.
     */
    public static Axis fromName(final String name) {
        return Arrays.stream(Axis.values())
                .filter(ax -> ax.getName().equalsIgnoreCase(name))
                .findFirst()
                .orElse(null);
    }

    /**
     * Find the Axis value having the given PLC axis number.
     * @param axnum The PLC axis number to find.
     * @return The corresponding Axis value, or null if no match was found.
     */
    public static Axis fromAxisNum(final int axnum) {
        return Arrays.stream(Axis.values())
                .filter(ax -> ax.getPLCAxisNum() == axnum)
                .findFirst()
                .orElse(null);
    }

    /**
     * Tells whether the axis is for the +X blade set or the -X.
     * @return true if it's the axis for the +X blade set.
     */
    public boolean isPlusXSide() {return plusXside;}

    /**
     * Gets the axis that's controlling the blade set on the +X side.
     * @return The axis.
     */
    public static Axis getPlusXSide() {return AXIS0.isPlusXSide() ? AXIS0 : AXIS1;}

    /**
     * Gets the axis that's controlling the blade set on the -X side.
     * @return The axis.
     */
    public static Axis getMinusXSide() {return AXIS0.isPlusXSide() ? AXIS1 : AXIS0;}


}
