package org.lsst.ccs.subsystems.shutter.sim;

import java.util.Random;

import java.util.function.Function;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;

/**
 * Makes a BladePosition from the information in a MotorPosition.
 * 
 * <p> At present this just takes the relative position and time, adding a random truncated
 * Gaussian jitter of standard deviation 0.0001 to the relative position.  (see {@link org.lsst.ccs.subsystems.shutter.common.BladePosition}).
 * The absolute position, which uses actual length units, is set to zero.
 * 
 * @author azemoon
 * @author tether
 */
public class MotorEncoderSimulator implements Function<MotorPosition, BladePosition> {

    /**
     * The standard deviation of the jitter added to the relative position.
     * For a full travel range of about 1 meter this would be about 1/10 mm.
     */
    public final static double JITTER_STDDEV = 0.0001;
    
    /**
     * The maximum permitted absolute value of the added jitter, i.e., where we
     * cut off the tails of the Gaussian.
     */
    public final static double MAX_JITTER = 5.0 * JITTER_STDDEV;
    
    private final Random random = new Random();

    @Override
    public BladePosition apply(final MotorPosition p) {
        double jitter = random.nextGaussian() * JITTER_STDDEV;
        if (Math.abs(jitter) > MAX_JITTER) jitter = MAX_JITTER * Math.signum(jitter);
        final double position = p.getPosition() + random.nextGaussian() * JITTER_STDDEV;
        final BladePosition bladePosition = new BladePosition(
            p.getTime(),
            0.0, // Absolute position not simulated.
            position);
        return bladePosition;
    }
}
