package org.lsst.ccs.subsystems.shutter.sim;

import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.lsst.ccs.subsystems.shutter.common.HallTransition;

/**
 * Calculates when each Hall sensor undergoes a transition from open to closed and back to open.
 * 
 * <p>Each instance of this class calculates a time-ordered set of Hall transitions for a single
 * motion of a blade set. The transitions may be recovered as many times as
 * needed by creating a stream from them.
 * 
 * <p>Instances of this class are immutable.
 * 
 * <p> This version of the class describes the old shutter test bench having a single blade set.
 * The positions of Hall sensors and magnets is completely hard-coded. Use of a blade-set configuration
 * is not yet implemented so the true positions of transitions in mm isn't available.
 * 
 * <p> Hall sensor model:
 * <ul>
 * <li>Relative blade set positions are used.
 * See {@link org.lsst.ccs.subsystems.shutter.common.BladePosition}.</li>
 * <li>Each magnet and sensor occupies zero width along the track.</li>
 * <li>There are nm magnets with spacing s mounted on the blade set, with the first magnet right
     on the edge.</li>
 * <li>There are nh Hall sensors with spacing nm*s mounted along the track. Each sensor is
 *     sensitive to a magnet within a window of length w/2 on each side of the sensor.
 *     The window of the first sensor starts at offset woff from zero and the window
 *     of the last sensor ends at 1 - woff. w/2 is less than s.</li>
 * </ul>
 * <p>When extending a blade set from the fully retracted position, the first transition pair
 * begins when the edge reaches woff with one further transition pair every time it travels
 * a further distance s. At the start of the last transition pair the last magnet will
 * be just at the start of the window of the last sensor at position 1 - woff - w. The
 * first magnet will therefore be at 1 - woff - w + (nm-1)*s and will have traveled
 * a distance of 1 - woff - w + (nm-1)*s - woff since the start of the first transition pair.
 * Since transition pairs are spaced at travel distance s the total distance traveled
 * must also be (no. of transitions - 1)*s = (nm*nh - 1)*s. Then 1 - 2*woff -w +(nm-1)*s =
 * (nm*nh-1)*s, or s = (1 - 2*woff - w) / (nm*(nh - 1)).
 * 
 * <p>Parameters, currently hard coded:
 * <ul>
 * <li>Nine magnets</li>
 * <li>Four Hall sensors</li>
 * <li>Sensor window offset = 0.0125</li>
 * <li>Sensor window size = 0.00625</li>
 * <li>Spacing between transition pairs = s = magnet spacing = 0.0359 (approx)</li>
 * <li>Spacing between sensors = nm*s = 0.3229 (approx)</li>
 * </ul>
 * 
 * <p>Since we know the position of the blade set's edge for every possible transition we
 * just need to pick those positions that fall within the path traced by the edge. Then we can
 * use the inverse position function of the motion profile to determine the time of each
 * transition.
 * @author azemoon
 * @author tether
 */
public class HallSensorSimulator {
    
    private static final int NUM_MAGNETS = 9;
    
    private static final int NUM_SENSORS = 4;
    
    private static final double SENSOR_WINDOW_OFFSET = 10.0 /800.0;
    
    private static final double SENSOR_WINDOW_SIZE = 5.0 / 800.0;
    
    private static final double TRANSITION_PAIR_SPACING =
            (1.0 - 2 * SENSOR_WINDOW_OFFSET - SENSOR_WINDOW_SIZE) / (NUM_MAGNETS*(NUM_SENSORS-1));

    private static class SensorData {
        public final int sensorId;
        public final boolean firstOfPair;
        public final double position;
        public long time;
        public SensorData(int id, boolean first, double pos) {
            this.sensorId = id;
            this.firstOfPair = first;
            this.position = pos;
        }
    }
    
    private final HallTransition[] transitions;
    
    /**
     * Find all the Hall transitions relevant to the given motion of the blade set.
     * @param startTime the starting time (a system time stamp in microseconds)
     * @param startPosition the starting position of the blade set edge
     * @param endPosition the ending position of the blade set edge
     * @param profile the description of the physical motion
     */
    public HallSensorSimulator(
            final long   startTime,
            final double startPosition,
            final double endPosition,
            final MotionProfile profile)
    {
        final int NUM_TRANSITIONS = 2 * NUM_MAGNETS * NUM_SENSORS;
        final boolean retracting = endPosition < startPosition;
        final double top = Math.max(startPosition, endPosition);
        final double bottom = Math.min(startPosition, endPosition);
        
        // Generate all possible transitions ordered according to
        // the direction of motion. Generate the transition numbering
        // first then build a transition for each number.
        IntStream indexes;
        if (!retracting)
            indexes = IntStream.range(0, NUM_TRANSITIONS);
        else
            indexes = IntStream.iterate(NUM_TRANSITIONS - 1, i -> i - 1).limit(NUM_TRANSITIONS);
        transitions = indexes
            .mapToObj(itrans -> {
                final int sensorId = itrans / NUM_MAGNETS / 2;
                final int ipair = itrans >> 1;
                final boolean even = (itrans & 1) == 0;
                final boolean firstOfPair = even ^ retracting; // Flip order when retracting.
                final double pos =
                        SENSOR_WINDOW_OFFSET +
                        ipair * TRANSITION_PAIR_SPACING +
                        (even ? 0.0 : SENSOR_WINDOW_SIZE);
                return new SensorData(sensorId, firstOfPair, pos);
            })
            // Keep transitions that lie within the range of motion.
            .filter(data -> (data.position <= top) && (data.position >= bottom))
             // Turn the transition position into a time-stamped HallTransition.
            .map(data -> {
                final double pos = data.position - startPosition;
                final long microsecs = startTime + Math.round(1e6 * profile.inverseDistance(pos));
                return new HallTransition(
                    microsecs,
                    data.sensorId,
                    !data.firstOfPair,  // Leading transition of each pair has sensor closed.
                    retracting,
                    0.0,                // Position in mm not available.
                    data.position);
            })
            .toArray(HallTransition[]::new);
    }
    
    /**
     * Gets the time-ordered stream of Hall transitions.
     * @return the transition stream
     */
    public Stream<HallTransition> getTransitions() {return Stream.of(transitions);}
}
