package org.lsst.ccs.subsystems.shutter.common;

import java.util.List;

/**
 * The interface to software controlling a single blade set, real or simulated.
 * @author tether
 */
public interface BladeSet {

    /**
     * Gets the current position of the BladeSet. May cause hardware operations.
     *
     * @return  the current relative position of the blade set
     * @see BladePosition
     *
     */
    public double getRelativePosition();


    /**
     * Moves the blade set to the specified position. May cause hardware operations.
     * @param position  the desired relative position of the leading edge of the blade set
     * @param moveTime  the move duration in seconds
     * @return a MovementHistory instance resulting from the motion
     * 
     * <p>Upon return getCurrentPosition() will reflect the final position of the move in any thread.
     *
     * @throws IllegalArgumentException if position is not valid
     *
     */
    public MovementHistory moveToPosition(double position, double moveTime);

    /**
     * Gets the side of the shutter to which the blade set belongs.
     */
    public ShutterSide getSide();

    /**
     * Gets the configuration of the BladeSet.
     *
     * @return  the BladeSet configuration
     *
     */
    public BladeSetConfiguration getBladeSetConfiguration();

    /**
     * Converts a motor position encoder value to an absolute position in mm of the leading
     * edge of the blade set on the camera X axis. 
     * 
     * @param encoderValue the value read from the encoder.
     * @return the absolute position in mm.
     */
    public double encoderToAbsolute(double encoderValue);

    /**
     * The inverse of {@link #encoderToAbsolute(double)}.
     * @param absolute the absolute position.
     * @return the corresponding encoder value.
     */
    public double absoluteToEncoder(double absolute);
    
    /**
     * Converts an absolute position along the camera X axis to a relative
     * blade set position in the range [0.0, 1.0], where 0.0 means fully retracted
     * (open) and 1.0 means fully extended (closed). The formula used is
     * {@code relative = (absolute - absoluteOpen) / (absoluteClosed - absoluteOpen)}.
     * @param abs the absolute position.
     * @return the relative position.
     * @see BladeSetConfiguration
     */
    public double absoluteToRelative(double abs);
    
    /**
     * The inverse of {@link #absoluteToRelative(double)}.
     * @param rel the relative position.
     * @return the absolute position.
     * @see BladeSetConfiguration
     */
    public double relativeToAbsolute(double rel);
    
    /**
     * Is the blade set fully retracted? That is, is its absolute position within the
     * shutter tolerance of the open position?
     * @return the answer.
     * @see ShutterConfiguration
     * @see BladeSetConfiguration
     */
    public boolean isFullyRetracted();
    
    /**
     * Is the blade set fully extended? That is, is its absolute position within the
     * shutter tolerance of the closed position?
     * @return the answer.
     * @see ShutterConfiguration
     * @see BladeSetConfiguration
     */
    public boolean isFullyExtended();

}
