package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;

/**
 *  Manages refrigeration cryo compressor limits.
 *
 *  @author Owen Saxton
 */
public class CryoCompLimits extends CompLimits {

    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "Celsius", description = "Phase separator temperature limit causing delayed compressor shutoff")
    private volatile double phaseSepTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "second", description = "Time delay before excess phase separator temperature causes compressor shutoff")
    private volatile int phaseSepTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "psia", description = "Discharge pressure difference limit inhibiting compressor startup")
    private volatile double pressDiffImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=false, units = "Celsius", description = "Low discharge temperature limit inhibiting compressor startup")
    private volatile double discTempLowImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "mm", description = "Low oil level limit causing immediate compressor shutoff")
    private volatile double oilLevelImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "Celsius", description = "Discharge temperature limit causing heater shutoff")
    private volatile double heaterTempLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "psia", description = "Discharge pressure above which orifice valve is closed")
    private volatile double orificeOffPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "psia", description = "Discharge pressure below which orifice valve is opened")
    private volatile double orificeOnPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "psia", description = "Discharge pressure above which filter dryer warning is issued")
    private volatile double dryerCheckHighPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true, units = "psia", description = "Suction pressure below which filter dryer warning is issued")
    private volatile double dryerCheckLowPress;


    /**
     *  Initializes cryo compressor limits.
     */
    @Override
    public void postInit()
    {
        super.postInit();
        limitData[CompSwConds.SWC_DISC_PRESS].delayTime = -1;
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].immedLimit = Double.NaN;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].isLower = true;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].delayLimit = Double.NaN;
        limitData[CompSwConds.SWC_PRESS_DIFF].delayLimit = Double.NaN;
        limitData[CompSwConds.SWC_OIL_LEVEL].isLower = true;
        limitData[CompSwConds.SWC_OIL_LEVEL].delayLimit = Double.NaN;
    }


    /**
     *  Sets the phase separator temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayLimit(double value)
    {
        phaseSepTempDelayLimit = value;
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].delayLimit = value;
    }


    /**
     *  Sets the phase separator temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayTime(int value)
    {
        phaseSepTempDelayTime = value;
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the discharge temperature low immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempLowImmedLimit(double value)
    {
        discTempLowImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].immedLimit = value;
    }


    /**
     *  Sets the discharge pressure difference immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPressDiffImmedLimit(double value)
    {
        pressDiffImmedLimit = value;
        limitData[CompSwConds.SWC_PRESS_DIFF].immedLimit = value;
    }


    /**
     *  Sets the oil level immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setOilLevelImmedLimit(double value)
    {
        oilLevelImmedLimit = value;
        limitData[CompSwConds.SWC_OIL_LEVEL].immedLimit = value;
    }


    /**
     *  Sets the heater temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setHeaterTempLimit(double value)
    {
        heaterTempLimit = value;
        limitData[CompSwConds.SWC_HEATER_TEMP].immedLimit = value;
    }


    /**
     *  Sets the orifice valve off pressure.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setOrificeOffPress(double value)
    {
        orificeOffPress = value;
        limitData[CompSwConds.SWC_ORF_OFF_PRESS].immedLimit = value;
    }


    /**
     *  Sets the orifice valve on pressure.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setOrificeOnPress(double value)
    {
        orificeOnPress = value;
        limitData[CompSwConds.SWC_ORF_ON_PRESS].immedLimit = value;
    }


    /**
     *  Sets the check filter dryers high pressure.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDryerCheckHighPress(double value)
    {
        dryerCheckHighPress = value;
        limitData[CompSwConds.SWC_DRYER_HI_PRESS].immedLimit = value;
    }


    /**
     *  Sets the check filter dryers low pressure.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDryerCheckLowPress(double value)
    {
        dryerCheckLowPress = value;
        limitData[CompSwConds.SWC_DRYER_LO_PRESS].immedLimit = value;
    }

}
