package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.RefrigAction;
import org.lsst.ccs.subsystem.refrig.constants.ChillerAlerts;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
//import org.lsst.ccs.subsystem.vacuum.constants.VacuumAgentProperties;

/**
 *  Listen for alerts from specified other subsystems; for a safety-related
 *  alert which at alarm level is meant to shut off chiller flow, raise a
 *  corresponding chiller-subsystem aler.
 */

public class ChillerAlertListener implements AlertListener, HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private ChillerSubsystem subsys;

    @LookupName
    private String name;

    private static final Logger LOG = Logger.getLogger(ChillerSubsystem.class.getName());

   /**
    *  For each listened-for Alert, this inner class holds the source and 
    *  expected payload, plus the Chiller Alert to be raised in response.
    */
    private static class AlertAction {

        //private String type;                 // an agent property
        private String source;               // originating subsystem
        private RefrigAction.Action action;  // expected payload
        private ChillerAlerts response;

        AlertAction(String source, RefrigAction.Action action, 
                    ChillerAlerts calert) {
            //this.type = type;
            this.source = source;
            this.action = action;
            this.response = calert;
        }

        //public String getType() {return type;}
        public String getSource() {return source;}
        public RefrigAction.Action getAction() {return action;}
        public ChillerAlerts getResponse() {return response;}
    }

    @ConfigurationParameter(isFinal = true, units = "unitless",
        description = "Subsystem listened to for vacuum alerts")
	protected volatile String vacuumAlertSource;

    @ConfigurationParameter(isFinal = true, units = "unitless",
        description = "Subsystem listened to for temperature alerts")
	protected volatile String thermalAlertSource;

    private List<String> sources = new ArrayList<>();
    //private List<String> types = new ArrayList<>();
    private List<AlertAction> actions = new ArrayList<>();
    // Map of Ids of Alerts received at Alarm level for each action
    private Map<AlertAction, List<String>> idMap = new HashMap<>();

   /**
    *  Init phase.
    */
    @Override
    public void init()
    {
        propertiesService.setAgentProperty(RefrigAction.getManagerName(), "");

        // If chiller not connected to camera, do not listen for alerts.
        if (!subsys.getConnectedToCamera()) {return;}

        // Listened-for alerts and actions; may later be configured
        actions.add(new AlertAction(vacuumAlertSource,
                                    RefrigAction.Action.VACUUM_INSUFFICIENT,
                                    ChillerAlerts.VACUUM_LIMIT));
        actions.add(new AlertAction(thermalAlertSource,
                                    RefrigAction.Action.COLD_PLATE_TOO_COLD,
                                    ChillerAlerts.COLD_TEMP_LOW));
        actions.add(new AlertAction(thermalAlertSource,
                                    RefrigAction.Action.COLD_PLATE_TOO_WARM,
                                    ChillerAlerts.COLD_TEMP_HIGH));

        for (AlertAction act : actions) {
            // List sources
            String src = act.getSource();
            if (!sources.contains(src)) sources.add(src);
            // Initialize idMap
            idMap.put(act, new CopyOnWriteArrayList<>());
        }

        // Start alert listening for specified sources
        alertService.startStatusAlertListening((info) -> {
	    return sources.contains(info.getName());
	});
        LOG.log(Level.INFO, name + " listening to Alerts from " + sources);
    }

   /**
    *  Start phase.
    */
    @Override
    public void start()
    {
         alertService.addListener(this);
    }

   /**
    *  Shutdown phase.
    */
    @Override
    public void shutdown()
    {
        //Remove alert listeners
        alertService.removeListener(this);
    }

   /**
    *  Handler for received AlertEvent
    *
    *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event) {
        switch (event.getType()) {
 
	   /** 
            *  Raised Alert:  If ALARM level. identify action from payload,
            *  keep track of originating alert Id, and raise ChillerAlert
            */
            case ALERT_RAISED:
                if (event.getLevel() == AlertState.ALARM) {
                    AlertAction action =  findAction(event.getAlert());
                    if (action != null) {
		        raiseAlarm(event.getAlert(), action);
                    }
		}
                break;

	       /** 
                *  Cleared alerts: remove from Lists for each action; if a 
                *  List is emptied, corresponding ChillerAlert is set NOMINAL
                */
	    case ALERT_CLEARED:
                List<String> cleared = event.getClearedIds();
                for (AlertAction action : actions) {
                    if (idMap.get(action).isEmpty()) {
                        continue;
                    }
                    for (String id : cleared) {
                        idMap.get(action).remove(id);
                    }
                    if (idMap.get(action).isEmpty()) {
                        alertService.raiseAlert(action.getResponse().newAlert(),
                                                AlertState.NOMINAL,
                                                " source Alert(s) cleared");
                    }
                }
                break;

	   /**
            *  Source agant and Chiller agent have connected.  Look over
            *  AlerttHistory for all relevant source alerts, and raise
            *  corresponging Chiller Alert at Apprpriate level.
            */
	    case AGENT_CONNECTION:
                String srcConn = event.getSource();
                LOG.log(Level.INFO, "Agent " + srcConn + " connected");
                for (AlertAction act : actions) {
                    if (!act.getSource().equals(srcConn)) {
			continue;
		    }
                    boolean raised = !idMap.get(act).isEmpty();
                    idMap.get(act).clear();
                    for (RaisedAlertHistory hist : event.getSummary().getAllRaisedAlertHistories())  {
                        if ((hist.getHighestAlertState() == AlertState.ALARM) &&
                            (findAction(hist.getHighestAlert()) == act)) {
                            raiseAlarm(hist.getHighestAlert(), act);
                        }
                    }
                    if (raised && idMap.get(act).isEmpty()) {
                        alertService.raiseAlert(act.getResponse().newAlert(),
                                                AlertState.NOMINAL,
                                                " source Alert(s) cleared");
		    }
                }
                break;

	    case AGENT_DISCONNECTION:
                LOG.log(Level.WARNING, "Agent " + event.getSource() + 
                        " disconnected");
                break;

	    default:
                break;
	}
    }

    /*  Find AlertAction for provided Alert, or null if none  */

    private AlertAction findAction(Alert alert) {
        AlertAction action = null;
        for (AlertAction act : actions) {
            if (RefrigAction.getData(alert) != null) {
                if (RefrigAction.getData(alert).action == act.getAction()) {
                    action = act;
                    break;
                }
            }
        }
        return action;
    }
    
   /**
    *  Respond to received Alert by raising Chiller Alert at alarm level
    *
    *  @paYam  Alert        received Alert
    *  @param  AlertAction  Chiller acttion
    */
    public void raiseAlarm(Alert alert, AlertAction action) {
        String id = alert.getAlertId();
        if (!idMap.get(action).contains(id)) {
             idMap.get(action).add(id);
        }
        alertService.raiseAlert(action.getResponse().newAlert(), 
                                AlertState.ALARM, 
				action.getResponse().getDescription() +
                                ", from Alert " + id);
    }

}
