package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.refrig.constants.PcpConditions;
import org.lsst.ccs.subsystem.refrig.constants.PcpLatches;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitches;

/**
 *  Handles the simulated PCP PLC.
 *
 *  @author Owen Saxton
 */
public class SimPcpPlutoDevice extends PcpPlutoDevice {

    /**
     *  Constants
     */
    private static final List<Integer>[] clearList = new ArrayList[PcpLatches.NUM_LATCHES];
    static {
        for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
            clearList[cond] = new ArrayList<>();
        }
        clearList[PcpLatches.LATCH_COLD_TEMP_HIGH].add(PcpLatches.LATCH_COLD_TEMP_HIGH);
        clearList[PcpLatches.LATCH_COLD_TEMP_LOW].add(PcpLatches.LATCH_COLD_TEMP_LOW);
    }
    private static final Map<Integer, Integer> switchMap = new HashMap<>();
    static {
        switchMap.put(PcpSwitches.SW_ENABLE_HEAT_PS1, PcpConditions.COND_HEAT_PS1);
        switchMap.put(PcpSwitches.SW_ENABLE_HEAT_PS2, PcpConditions.COND_HEAT_PS2);
        switchMap.put(PcpSwitches.SW_ENABLE_HEAT_PS3, PcpConditions.COND_HEAT_PS3);
    }

    /**
     *  Data fields.
     */
    private static final Logger LOG = Logger.getLogger(SimPcpPlutoDevice.class.getName());
    private final boolean[] switches = new boolean[PcpSwitches.NUM_SWITCHES_ALL];
    private final Boolean[] latches = new Boolean[PcpLatches.NUM_LATCHES];
    private final boolean[] conditions = new boolean[PcpConditions.NUM_CONDITIONS];
    private final int[] simTempLimits = {-39, 39, -139, 29};


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
        LOG.log(Level.INFO, "Connected to {0} (simulated Pluto gateway)", getPath());
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(Channel ch)
    {
        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        return 315.0 + ch.getHwChan();
    }


    /**
     *  Tests whether the PLC is active
     *
     *  @return  Whether the PLC is active, or null if offline
     */
    @Override
    public Boolean isPlcActive()
    {
        return true;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw] = on;
        Integer cond = switchMap.get(sw);
        if (cond != null) {
            conditions[cond] = on;
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The state is not the state of the bit that was toggled, but is read back
     *  either directly from the controlled hardware, or from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return switches[sw];
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    @Override
    public Boolean isLatchActive(int cond)
    {
        Boolean latch = latches[cond];
        return latch != null && latch;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    @Override
    public Boolean isLatchLatched(int cond)
    {
        Boolean latch = latches[cond];
        return latch != null && !latch;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    @Override
    public void clearLatch(int cond)
    {
        for (int cnd : clearList[cond]) {
            Boolean latch = latches[cnd];
            latches[cnd] = latch != null && !latch ? null : latch;
        }
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    @Override
    public Boolean isConditionActive(int cond)
    {
        return conditions[cond];
    }


    /**
     *  Gets the temperature limits.
     *
     *  @return  The 4-element array of temperature limits
     */
    @Override
    public int[] getTempLimits()
    {
        return simTempLimits;
    }


    @Command(type = Command.CommandType.ACTION)
    public void setLatchActive(@Argument(description="Latched condition number") int cond) {
        latches[cond] = true;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setLatchLatched(@Argument(description="Latched condition number") int cond) {
        latches[cond] = false;
    }

    @Command(type = Command.CommandType.ACTION)
    public void setCondition(@Argument(description="Running condition number") int cond,
                             @Argument(description="Condition on state") boolean set) {
        conditions[cond] = set;
    }

}
