package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;

/**
 *  Persists simulated compressor Pluto PLC data.
 *
 *  @author Owen Saxton
 */
public class SimCompPlutoPersist implements HasLifecycle {

    /**
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private PersistencyService persistencyService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, SimCompPlutoDevice> devices = new HashMap<>();

    @Persist
    private volatile Map<String, LatchState[]> latchesMap = new HashMap<>();
    @Persist
    private volatile Map<String, ConditionState[]> conditionsMap = new HashMap<>();
    @Persist
    private volatile Map<String, boolean[]> switchesMap = new HashMap<>();

    /**
     *  Build initialization
     */
    @Override
    public void build()
    {
        // Configure persistency service
        persistencyService.setAutomatic(true, true);
        persistencyService.setPersistencyFileName(subsys.getName() + "_" + name + ".properties");
    }

    /**
     *  Initialization
     */
    @Override
    public void postStart()
    {
        for (String devcName : devices.keySet()) {
            String compName = devcName.split("/")[0];
            SimCompPlutoDevice device = devices.get(devcName);
            LatchState[] latches = latchesMap.get(compName);
            if (latches != null) {
                device.setLatches(latches);
            }
            latchesMap.put(compName, device.getLatches());
            ConditionState[] conditions = conditionsMap.get(compName);
            if (conditions != null) {
                device.setConditions(conditions);
            }
            conditionsMap.put(compName, device.getConditions());
            boolean[] switches = switchesMap.get(compName);
            if (switches != null) {
                device.setSwitches(switches);
            }
            switchesMap.put(compName, device.getSwitches());
        }
    }

}
