package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.refrig.constants.PcpHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.PcpHeaters;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.MonitorControl;
import org.lsst.ccs.subsystem.refrig.constants.PLCState;
import org.lsst.ccs.subsystem.refrig.constants.PcpAlert;
import org.lsst.ccs.subsystem.refrig.constants.PcpConditions;
import org.lsst.ccs.subsystem.refrig.constants.PcpLatches;
import org.lsst.ccs.subsystem.refrig.constants.PcpLimits;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitchState;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitches;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.data.PcpSysState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Enables testing of the prototype cold plate.
 *
 *  @author Owen Saxton
 */
public class PrototypeColdPlateSubsystem extends Subsystem implements HasLifecycle, AlertListener {

    private static final Map<Integer, PcpAlert> alertMap = new HashMap<>();
    static {
        alertMap.put(PcpLatches.LATCH_COLD_TEMP_HIGH, PcpAlert.COLD_TEMP_HIGH);
        alertMap.put(PcpLatches.LATCH_COLD_TEMP_LOW, PcpAlert.COLD_TEMP_LOW);
        alertMap.put(PcpLatches.LATCH_SMOKE_DETC, PcpAlert.SMOKE_DETC);
    }
    private static final Map<String, String> revAlertMap = new HashMap<>();
    static {
        for (Map.Entry e : alertMap.entrySet()) {
            revAlertMap.put(((PcpAlert)e.getValue()).getId(), PcpLatches.getName((int)e.getKey()));
        }
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private PcpPlutoDevice plutoDevc;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private PowerDevice powerDevice;

    private int[] powerChans;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(PrototypeColdPlateSubsystem.class.getName());
    private final PcpSysState pcpState = new PcpSysState();
    private MonitorTaskControl monitorControl;
    private boolean running = false, gotCommand = false;
    private boolean[] powerInited;


    /**
     *  Constructor.
     */
    public PrototypeColdPlateSubsystem() {
        super("prototype-cold-plate", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        //Create and schedule an AgentPeriodicTask to update the PCP state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("pcp-state", () -> updatePcpState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     * Init phase.
     * 
     * Register Alerts raised by this subsystem.
     */
    @Override
    public void init()
    {
        // Register all possible alerts
        for (PcpAlert alert: PcpAlert.values()) {
            alertService.registerAlert(alert.getAlert());
        }
    }


    /**
     *  Initializes the subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a protection subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.PCP_TYPE, PrototypeColdPlateSubsystem.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        if (powerChans == null) {
            LOG.info("No heater channels specified");
        }
        else if (powerChans.length != PcpHeaters.NUM_HEATERS) {
            ErrorUtils.reportConfigError(LOG, name, "powerChans", "does not contain exactly 3 items");
        }
        powerInited = new boolean[powerChans.length];

        if (plutoDevc != null) {
            for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
                pcpState.setLatch(cond, LatchState.CLEAR);
            }
            for (int cond = 0; cond < PcpConditions.NUM_CONDITIONS; cond++) {
                pcpState.setCondition(cond, ConditionState.NO);
            }
        }
        else {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce the system
        LOG.info("Prototype cold plate subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the PCP system.
     *
     *  @return  The PCP system state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the PCP system state")
    public PcpSysState getSystemState()
    {
        pcpState.setTickMillis(monitorControl.getFastPeriod());
        return pcpState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names")
    public List<String> getSwitchNames()
    {
        return PcpSwitches.getNames();
    }


    /**
     *  Turns a (named) switch on or off.
     *
     *  @param  swch  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String swch,
                                 @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        gotCommand = true;
        int sw = PcpSwitches.getId(swch);
        PcpSwitchState state = pcpState.getSwitchState(sw);
        if (state == PcpSwitchState.OFFLINE) return;
        plutoDevc.setSwitchOn(sw, on);
    }


    /**
     *  Gets the list of latched condition names.
     *
     *  @return  The condition names.
     */
    @Command(type=Command.CommandType.QUERY, description="Get latched condition names")
    public List<String> getLatchNames()
    {
        return PcpLatches.getNames();
    }


    /**
     *  Clears a (named) latched condition.
     *
     *  @param  cond  The condition name.
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a latched condition")
    public void clearLatch(@Argument(description="The condition name") String cond) throws RefrigException
    {
        gotCommand = true;
        plutoDevc.clearLatch(PcpLatches.getId(cond));
    }


    /**
     *  Gets the list of heater names.
     *
     *  @return  The heater names.
     */
    @Command(type=Command.CommandType.QUERY, description="Get heater names")
    public List<String> getHeaterNames()
    {
        return PcpHeaters.getNames();
    }


    /**
     *  Sets a heater state.
     *
     *  @param  htrName  The heater name
     *  @param  on   Whether to set on
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a heater power enabled state")
    public void setHeaterState(@Argument(description="The heater name") String htrName,
                               @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        gotCommand = true;
        int htr = PcpHeaters.getId(htrName);
        synchronized(pcpState) {
            PcpHeaterState oldState = pcpState.getHeaterState(htr);
            if (oldState != PcpHeaterState.OFFLINE && oldState != PcpHeaterState.DISABLD) {
                PcpHeaterState newState = on ? PcpHeaterState.ON : PcpHeaterState.OFF;
                if (newState != oldState) {
                    pcpState.setHeaterState(htr, newState);
                    setHeaterPower(htr);
                }
            }
        }
    }


    /**
     *  Sets a heater power value.
     *
     *  @param  htrName  The heater name
     *  @param  value  The power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a heater power set point")
    public void setHeaterPower(@Argument(description="The heater name") String htrName,
                               @Argument(description="The power set point") double value) throws RefrigException
    {
        gotCommand = true;
        int htr = PcpHeaters.getId(htrName);
        synchronized(pcpState) {
            pcpState.setHeaterPower(htr, value);
            setHeaterPower(htr);
        }
    }


    /**
     *  Sets the power for a heater.
     */
    private void setHeaterPower(int htr)
    {
        int chan = powerChans[htr];
        if (PcpHeaterState.isOnState(pcpState.getHeaterState(htr))) {
            powerDevice.enableOutput(chan, true);
            powerDevice.setPower(chan, pcpState.getHeaterPower(htr));
        }
        else {
            powerDevice.enableOutput(chan, false);
        }
    }


    /**
     *  Updates the protection system state periodically.
     *
     *  The protection state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     */
    private void updatePcpState()
    {
        if (!running) return;

        boolean changed = monitorControl.hasPeriodChanged();

        synchronized(pcpState) {

            Boolean plcActive = plutoDevc.isPlcActive();
            PLCState newState = plcActive == null ? PLCState.OFFLINE : plcActive ? PLCState.ALIVE : PLCState.DEAD;
            if (newState != pcpState.getPlcState()) {
                changed = true;
                pcpState.setPlcState(newState);
                PcpAlert alert = PcpAlert.CHILLER_PLC_NOT_ALIVE;
                String plcDesc = "PCP protection PLC ";
                if (newState == PLCState.ALIVE) {
                    lowerAlert(alert, plcDesc + "is alive");
                }
                else {
                    String errDesc = newState == PLCState.DEAD ? "has died: error = " + plutoDevc.getErrorCode() : "is offline"; 
                    raiseAlert(alert, plcDesc + errDesc);
                }
            }

            for (int sw = 0; sw < PcpSwitches.NUM_SWITCHES; sw++) {
                PcpSwitchState state = plutoDevc.getSwitchState(sw);
                if (state != pcpState.getSwitchState(sw)) {
                    pcpState.setSwitchState(sw, state);
                    changed = true;
                }
            }

            for (int cond = 0; cond < PcpLatches.NUM_LATCHES; cond++) {
                Boolean active = plutoDevc.isLatchActive(cond);
                Boolean latched = plutoDevc.isLatchLatched(cond);
                LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                                   latched ? LatchState.LATCHED :
                                   active ? LatchState.ACTIVE : LatchState.CLEAR;
                PcpAlert alert = alertMap.get(cond);
                LatchState oldState = pcpState.getLatch(cond); 
                if (state != oldState) {
                    pcpState.setLatch(cond, state);
                    if (state == LatchState.ACTIVE) {
                        raiseAlert(alert, "Protection PLC error condition set");
                    }
                    else if (state != LatchState.OFFLINE) {
                        if (oldState != LatchState.ACTIVE && state == LatchState.LATCHED) {
                            raiseAlert(alert, "Protection PLC error condition set");
                        }
                        if (oldState == LatchState.ACTIVE || state == LatchState.LATCHED) {
                            lowerAlert(alert, "Protection PLC error condition cleared");
                        }
                    }
                    changed = true;
                }
            }

            for (int cond = 0; cond < PcpConditions.NUM_CONDITIONS; cond++) {
                Boolean active = plutoDevc.isConditionActive(cond);
                ConditionState state = active == null ? ConditionState.OFF :
                                       active ? ConditionState.YES : ConditionState.NO;
                if (state != pcpState.getCondition(cond)) {
                    pcpState.setCondition(cond, state);
                    changed = true;
                }
            }

            if (pcpState.getLimit(0) == Integer.MAX_VALUE) {
                int[] limits = plutoDevc.getTempLimits();
                for (int j = 0; j < PcpLimits.NUM_LIMITS; j++) {
                    pcpState.setLimit(j, limits[j]);
                }
                changed |= pcpState.getLimit(0) != Integer.MAX_VALUE;
            }

            for (int htr = 0; htr < PcpHeaters.NUM_HEATERS; htr++) {
                PcpHeaterState oldState = pcpState.getHeaterState(htr);
                int chan = powerChans[htr];
                PcpHeaterState state = !powerDevice.isOnline() ? PcpHeaterState.OFFLINE :
                                       !powerDevice.isEnabled(chan) ? PcpHeaterState.OFF :
                                       powerDevice.hasVoltError(chan) ? PcpHeaterState.VOLTERR : 
                                       powerDevice.hasNoLoad(chan) ? PcpHeaterState.NOLOAD :
                                       PcpHeaterState.ON;

                if (state != oldState) {
                    pcpState.setHeaterState(htr, state);
                    if (oldState == PcpHeaterState.OFFLINE) {
                        if (!powerInited[htr]) {
                            if (powerDevice.isEnabled(chan)) {
                                double power = powerDevice.getPower(chan);
                                pcpState.setHeaterPower(htr, power);
                            }
                            powerInited[htr] = true;
                        }
                    }
                    setHeaterPower(htr);
                    changed = true;
                }
            }
        }

        if (gotCommand) {
            changed = true;
            gotCommand = false;
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The protection alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(PcpAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The protection alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(PcpAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            String cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (RefrigException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     *  Publishes the state of the PCP system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        pcpState.setTickMillis(monitorControl.getFastPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(PcpSysState.KEY, pcpState));
    }
    
}
