package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Implements a simple simulated device.
 *
 *  @author Owen Saxton
 */
public class SimpleSimDevice extends Device {

    private final Map<String, Integer> channelMap = new HashMap<>();
    private final List<Double> channelValues = new ArrayList<>();


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The channel to check;
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(Channel ch)
    {
        int index = channelValues.size();
        channelMap.put(ch.getName(), index);
        channelValues.add((double)index);
        return new int[]{index, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  ch   The Channel to read.
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        return channelValues.get(ch.getType());
    }


    /**
     *  Command to set a channel value.
     *
     *  @param  chan   The channel name.
     *  @param  value  The channel value
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws RefrigException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new RefrigException("Invalid channel name");
        }
        channelValues.set(index, value);
    }


    /**
     *  Command to get the list of channel names.
     *
     *  @return  The channel names, as a Set
     */
    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
