package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.GP390;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles an MKS GP390 vacuum gauge.
 *
 *  @author Owen Saxton
 */
public class GP390Device extends Device {

    /**
     *  Data fields.
     */
    private final static int BAUD_RATE = 19200;

    private String  devcName; // Serial device name
    private int     address;  // RS-485 address of gauge

    private static final Logger LOG = Logger.getLogger(GP390Device.class.getName());
    private final GP390 gp = new GP390();  // Associated GP390 object
    private boolean initError = false;


    /**
     *  Constructor.
     *
     *  @param  devcName  The serial device name
     *  @param  addr      The RS-485 node address
     */
    public GP390Device(String devcName, int addr)
    {
        this.devcName = devcName;
        address = addr;
    }


    public GP390Device() {
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (devcName == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "serial", "is missing");
        }
        fullName = "GP390 vacuum gauge (" + devcName + ":" + address + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            gp.open(GP390.ConnType.SERIAL, devcName, BAUD_RATE, address);
            gp.setLock(false);
            gp.setPressureUnit(GP390.UNIT_TORR);
            gp.setLock(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                gp.close();
            }
            catch (DriverException ce) {
                // Can happen if open failed
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            gp.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  ch   The Channel to read.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(Channel ch)
    {
        if (isOnline()) {
            try {
                return gp.readPressure();
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading pressure: {0}", e);
                setOnline(false);
            }
        }
        return Double.NaN;
    }

}
