package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Implements a simple simulated device.
 *
 *  @author Owen Saxton
 */
public class SimpleSimDevice extends Device {

    private final Map<String, Integer> channelMap = new HashMap<>();
    private final List<Double> channelValues = new ArrayList<>();


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
    {
        int index = channelValues.size();
        channelMap.put(name, index);
        channelValues.add((double)index);
        return new int[]{index, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return channelValues.get(type);
    }


    /**
     *  Command to set a channel value.
     *
     *  @param  chan   The channel name.
     *  @param  value  The channel value
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws RefrigException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new RefrigException("Invalid channel name");
        }
        channelValues.set(index, value);
    }


    /**
     *  Command to get the list of channel names.
     *
     *  @return  The channel names, as a Set
     */
    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
