package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Defines a power supply interface.
 *
 *  @author Owen Saxton
 */
public class PowerDevice extends Device implements HasLifecycle {
    
    /**
     *  Inner class for saving the power state of a channel.
     */
    public static class ChannelState {
        
        private double
            resistance,    // Load resistance
            power;         // Load power
        private boolean
            enabled,       // Output has been enabled
            voltError,     // Error setting the voltage
            noLoad;        // Unable to determine the load resistance

    }

    protected static final int
        NETWORK_CONN_INVALID = 0x01;

    protected static final int
        MON_TYPE_POWER = 0,
        MON_TYPE_VOLTAGE = 1,
        MON_TYPE_CURRENT = 2;

    private final static double
        MIN_VOLTAGE = 4,
        MAX_CURRENT = 3,
        TIMEOUT = 2.0;

    private static final String
        REFRIG = "Refrig",
        CONN_TYPE = "connType",
        DEVC_ID   = "devcId",
        DEVC_PARM = "devcParm";

    protected static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("POWER", MON_TYPE_POWER);
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @ConfigurationParameter(name=CONN_TYPE, category=REFRIG, isFinal=true)
    protected volatile DriverConstants.ConnType connType;
    @ConfigurationParameter(name=DEVC_ID, category=REFRIG, isFinal=true)
    protected volatile String devcId;
    @ConfigurationParameter(name=DEVC_PARM, category=REFRIG, isFinal=true)
    protected volatile int devcParm = 0;

    protected double[] maxCurrent;              // Hardware current limits
    protected double softMaxCurrent = 0.0;      // Software current limit
    protected double minVoltage = MIN_VOLTAGE;  // Probe voltage
    protected int settlingTime = 100;           // Wait time after setting voltage (ms)

    private static final Logger LOG = Logger.getLogger(PowerDevice.class.getName());
    private final String devcName;
    protected PowerSupplyDriver psd;
    private final int options, minChan, maxChan;
    private final double voltages[], currents[];
    private final Map<Integer, ChannelState> states = new HashMap<>();
    private boolean initError = false;


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  psd       The power supply device driver object
     *  @param  options   The options word
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    public PowerDevice(String name, PowerSupplyDriver psd, int options, int minChan, int maxChan)
    {
        super();
        this.devcName = name;
        this.psd = psd;
        this.options = options;
        this.minChan = minChan;
        this.maxChan = maxChan;
        maxCurrent = new double[maxChan - minChan + 1];
        Arrays.fill(maxCurrent, MAX_CURRENT);
        voltages = new double[maxChan - minChan + 1];
        currents = new double[maxChan - minChan + 1];
        for (int chan = minChan; chan <= maxChan; chan++) {
            states.put(chan, new ChannelState());
        }
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to maintain the set power values
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("maintain-power-" + name, () -> maintainPower()).withPeriod(Duration.ofMillis(6000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Gets the minimum channel number.
     *
     *  @return  The minimum channel number
     */
    public int getMinChannel()
    {
        return minChan;
    }


    /**
     *  Gets the maximum channel number.
     *
     *  @return  The maximum channel number
     */
    public int getMaxChannel()
    {
        return maxChan;
    }

   
    /**
     *  Gets the connection type.
     *
     *  @return  The connection type enum
     */
    @Command(type=Command.CommandType.QUERY, description="Gets the connection type")
    public DriverConstants.ConnType getConnType()
    {
        return connType;
    }


    /**
     *  Gets the device ID.
     *
     *  @return  The device identification string
     */
    @Command(type=Command.CommandType.QUERY, description="Gets the device ID")
    public String getDevcId()
    {
        return devcId;
    }


    /**
     *  Gets the device parameter.
     *
     *  @return  The device parameter
     */
    @Command(type=Command.CommandType.QUERY, description="Gets the device parameter")
    public int getDevcParm()
    {
        return devcParm;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (connType == null) {
            ErrorUtils.reportConfigError(LOG, name, "connType", "is missing");
        }
        if (connType == DriverConstants.ConnType.NET && (options & NETWORK_CONN_INVALID) != 0) {
            ErrorUtils.reportConfigError(LOG, getName(), "connType", "is invalid: " + connType);
        }
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = devcName + (devcId.isEmpty() ? "" : " (" + devcId + ")");
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            psd.open(connType, devcId, devcParm);
            initSensors();
            //psd.setTimeout(timeout);
            for (int j = 0; j < maxCurrent.length; j++) {
                setCurrent(minChan + j, maxCurrent[j]);
            }
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                psd.close();
            }
            catch (DriverException ce) {
                // Will happen if open was unsuccessful
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            psd.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < minChan || hwChan > maxChan) {
            ErrorUtils.reportChannelError(LOG, name, "HW channel", hwChan);
        }
        Integer iType = mTypeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        return new int[]{iType, 0};
    }


    /**
     *  Initializes voltages and currents as unread.
     */
    @Override
    protected void readChannelGroup()
    {
        Arrays.fill(voltages, Double.NaN);
        Arrays.fill(currents, Double.NaN);
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (isOnline()) {
            try {
                switch (type) {
                case MON_TYPE_CURRENT:
                    value = currents[hwChan - minChan] = readCurrent(hwChan);
                    break;

                case MON_TYPE_VOLTAGE:
                    value = voltages[hwChan - minChan] = readVoltage(hwChan);
                    break;

                case MON_TYPE_POWER:
                    value = voltages[hwChan - minChan] * currents[hwChan - minChan];
                    if (Double.isNaN(value)) {
                        value = readVoltage(hwChan) * readCurrent(hwChan);
                    }
                    break;
                }
            }
            catch (DriverException e) {
                logError("Error reading from", hwChan, e);
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Sets the online state.
     *
     *  If coming on line, set state data from reading the hardware.
     *
     *  @param  online  The online state to set: true or false
     */
    @Override
    protected void setOnline(boolean online)
    {
        super.setOnline(online);
        if (isOnline()) {
            
            //WHY IS THIS HERE? Can it be moved to initialize?
            for (int chan : states.keySet()) {
                ChannelState state = states.get(chan);
                try {
                    state.enabled = getOutput(chan);
                    state.power = Math.round(10 * readVoltage(chan) * readCurrent(chan)) / 10.0;
                    state.voltError = false;
                    state.noLoad = false;
                }
                catch (DriverException e) {
                    logError("Error reading from", chan, e);
                }
            }
        }
    }


    /**
     *  Enables the output.
     *
     *  @param  chan    The hardware channel
     *  @param  enable  The enabled state to set
     */
    public void enableOutput(int chan, boolean enable)
    {
        ChannelState state = getChannelState(chan);
        try {
            setOutput(chan, enable);
            state.enabled = enable;
            if (!enable) {
                state.voltError = false;
                state.noLoad = false;
            }
        }
        catch (DriverException e) {
            logError("Error enabling output for", chan, e);
        }
    }


    /**
     *  Sets the power.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The power to set, or NaN to use current value
     */
    public void setPower(int chan, double value)
    {
        ChannelState state = getChannelState(chan);
        synchronized(state) {
            if (!state.enabled) return;
            List<Float> values = new ArrayList<>();
            if (!Double.isNaN(value)) {
                state.power = value;
            }
            if (state.resistance == 0) {
                values.add((float)minVoltage);
                setChanVoltage(chan, minVoltage);
                try {
                    Thread.sleep(settlingTime);
                }
                catch (InterruptedException e) {}
                double amps = readChanCurrent(chan);
                values.add((float)amps);
                if (amps > 0) {
                    state.resistance = readChanVoltage(chan) / amps;
                    state.noLoad = false;
                }
                else {
                    state.resistance = 0.0;
                }
            }
            if (state.resistance != 0) {
                double volts = Math.sqrt(state.power * state.resistance);
                if (softMaxCurrent > 0.0) {
                    volts = state.resistance * Math.min(volts / state.resistance, softMaxCurrent);
                }
                setChanVoltage(chan, volts);
                values.add((float)volts);
                try {
                    Thread.sleep(settlingTime);
                }
                catch (InterruptedException e) {}
                double amps = readChanCurrent(chan);
                values.add((float)amps);
                state.resistance = (amps <= 0) ? 0 : readChanVoltage(chan) / amps;
                if (state.resistance != 0) {
                    volts = Math.sqrt(state.power * state.resistance);
                    if (softMaxCurrent > 0.0) {
                        volts = state.resistance * Math.min(volts / state.resistance, softMaxCurrent);
                    }
                    setChanVoltage(chan, volts);
                    values.add((float)volts);
                }
            }
            if (state.resistance == 0) {
                if (!state.noLoad) {
                    logError("Unable to determine load resistance for", chan, null);
                }
                state.noLoad = true;
            }
            if (LOG.isLoggable(Level.FINE)) {
                StringBuilder msg = new StringBuilder("Setting " + name + " channel " + chan + " power to ");
                msg.append((float)state.power).append(": voltages/currents =");
                for (Float v : values) {
                    msg.append(" ").append(v);
                }
                LOG.fine(msg.toString());
            }
        }
    }


    /**
     *  Maintains the power for all channels at their set value.
     */
    protected void maintainPower()
    {
        for (int chan : states.keySet()) {
            setPower(chan, Double.NaN);
        }
    }


    /**
     *  Gets the power for a channel.
     *
     *  @param  chan   The hardware channel
     *  @return  The power value
     */
    public double getPower(int chan)
    {
        return getChannelState(chan).power;
    }


    /**
     *  Gets the enabled state.
     *
     *  @param  chan   The hardware channel
     *  @return  Whether enabled
     */
    public boolean isEnabled(int chan)
    {
        return getChannelState(chan).enabled;
    }


    /**
     *  Gets whether a voltage setting error occurred.
     *
     *  @param  chan   The hardware channel
     *  @return  The error state
     */
    public boolean hasVoltError(int chan)
    {
        return getChannelState(chan).voltError;
    }


    /**
     *  Gets whether a "no load" error occurred.
     *
     *  @param  chan   The hardware channel
     *  @return  The error state
     */
    public boolean hasNoLoad(int chan)
    {
        return getChannelState(chan).noLoad;
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The voltage to set
     */
    private void setChanVoltage(int chan, double value)
    {
        if (!isOnline()) return;
        ChannelState state = getChannelState(chan);
        try {
            setVoltage(chan, value);
            state.voltError = false;
        }
        catch (DriverException e) {
            if (!state.voltError) {
                logError("Error setting voltage on", chan, e);
            }
            state.voltError = true;
        }
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chan  The hardware channel
     *  @return  The voltage
     */
    private double readChanVoltage(int chan)
    {
        if (!isOnline()) return Double.NaN;
        try {
            return readVoltage(chan);
        }
        catch (DriverException e) {
            logError("Error reading voltage from", chan, e);
            return Double.NaN;
        }
    }


    /**
     *  Reads the current.
     *
     *  @param  chan  The hardware channel
     *  @return  The current
     */
    private double readChanCurrent(int chan)
    {
        if (!isOnline()) return Double.NaN;
        try {
            return readCurrent(chan);
        }
        catch (DriverException e) {
            logError("Error reading current from", chan, e);
            return Double.NaN;
        }
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The voltage to set
     *  @throws  DriverException
     */
    public void setVoltage(int chan, double value) throws DriverException
    {
        psd.setVoltage(value, chan);
    }


    /**
     *  Sets the current.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The current to set
     *  @throws  DriverException
     */
    public void setCurrent(int chan, double value) throws DriverException
    {
        psd.setCurrent(value, chan);
    }


    /**
     *  Sets the output state.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The output state to set
     *  @throws  DriverException
     */
    public void setOutput(int chan, boolean value) throws DriverException
    {
        psd.setOutput(value, chan);
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chan  The hardware channel
     *  @return  The voltage
     *  @throws  DriverException
     */
    public double readVoltage(int chan) throws DriverException
    {
        return psd.readVoltage(chan);
    }


    /**
     *  Reads the current.
     *
     *  @param  chan  The hardware channel
     *  @return  The current
     *  @throws  DriverException
     */
    public double readCurrent(int chan) throws DriverException
    {
        return psd.readCurrent(chan);
    }


    /**
     *  Gets the output state.
     *
     *  @param  chan   The hardware channel
     *  @return  The output state
     *  @throws  DriverException
     */
    public boolean getOutput(int chan) throws DriverException
    {
        return psd.getOutput(chan);
    }


    /**
     *  Gets the state object for a channel.
     *
     *  @param  chan   The hardware channel
     *  @return  The channel state
     */
    private ChannelState getChannelState(int chan)
    {
        return states.get(chan);
    }


    /**
     *  Log an error message for a channel.
     *
     *  @param  msg   The message text
     *  @param  chan  The hardware channel
     *  @param  excp  The exception, or null if none
     */
    private void logError(String msg, int chan, DriverException e)
    {
        LOG.log(Level.SEVERE, "{0} {1} channel {2}{3}", new Object[]{msg, name, chan, e == null ? "" : ": " + e});
    }

}
