package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.common.data.MonitorTask;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.data.HexState;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;
import org.lsst.ccs.subsystem.refrig.data.UpdatePeriod;

/**
 *  Implements the refrigeration heat exchanger monitoring subsystem.
 *
 *  @author Owen Saxton
 */
public class HexMain extends Subsystem implements HasLifecycle, AgentPresenceListener, StatusMessageListener {

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @ConfigurationParameter(isFinal = true)
    private volatile double rtnToEvapOffset = 0.3;

    private static final Logger LOG = Logger.getLogger(HexMain.class.getName());
    private final HexState hexState = new HexState();
    private String refrigGroup;
    private MonitorTaskControl monitorControl;
    private Map<String, MonitorTask> monitorTaskMap;

    /**
     *  Constructor.
     */
    public HexMain() {
        super("hex", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {

        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, RefrigConstants.MONITOR_CONTROL);

        // Create and schedule an AgentPeriodicTask to publish the hex state when updated
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("hex-state", () -> publishUpdatedHexState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a heat exchanger subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.HEX_TYPE, HexMain.class.getCanonicalName());

        // Add an agent present listener
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        refrigGroup = RefrigUtils.getGroupName(subsys.getAgentInfo());

        // Initialize monitor task data
        monitorTaskMap = monitorControl.getMonitorTaskMap();
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Heat exchanger subsystem started");
    }


    /**
     *  Listens for the arrival of the companion refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                if (RefrigUtils.getGroupName(agent).equals(refrigGroup)) {
                    getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                                                   && msg instanceof StatusSubsystemData);
                    break;
                }
            }
        }
    }


    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE) && RefrigUtils.getGroupName(agent).equals(refrigGroup)) {
                getMessagingAccess().removeStatusMessageListener(this);
            }
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  Sets the tick period from the refrig system value.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData sd = (StatusSubsystemData)msg;
        if (sd.getDataKey().equals(UpdatePeriod.KEY)) {
            int tickMillis = ((UpdatePeriod)((KeyValueData)sd.getSubsystemData()).getValue()).getTickMillis();
            if (tickMillis != monitorControl.getPublishPeriod()) {
                monitorControl.setPublishPeriod(tickMillis);
            }
        }
    }


    /**
     *  Gets the state of the heat exchanger module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the heat exchanger state", level=0)
    public HexState getSystemState()
    {
        synchronized (hexState) {
            hexState.setTickMillis(monitorControl.getPublishPeriod());
            hexState.clearMonitorTasks();
            for (MonitorTask task : monitorTaskMap.values()) {
                hexState.addMonitorTask(task);
            }
            return hexState;
        }
    }    


    /**
     *  Publishes the updated heat exchanger state.
     *
     *  This is called regularly as a timer task.
     */
    private void publishUpdatedHexState()
    {
        synchronized (hexState) {
            boolean changed = monitorControl.hasPeriodChanged();
            hexState.clearMonitorTasks();
            for (MonitorTask task : monitorTaskMap.values()) {
                if (task.hasChanged()) {
                    hexState.addMonitorTask(task);
                    changed = true;
                }
            }
            if (changed) {
                hexState.setTickMillis(monitorControl.getPublishPeriod());
                publishSubsystemDataOnStatusBus(new KeyValueData(HexState.KEY, hexState));
            }
        }
    }


    double getRtnToEvapOffset() {
        return rtnToEvapOffset;
    }
    
}
