package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CryoCompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Controls a refrigeration cryo compressor.
 *
 *  @author Owen Saxton
 */
public class CryoCompressor extends Compressor {

    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private CompMaq20PWMControl pwmControl;

    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double phaseSepTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int phaseSepTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double heaterTempLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double orificeOffPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double orificeOnPress;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double pressDiffImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double discTempLowImmedLimit;

    private String suctPressChan, phaseSepTempChan;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(CryoCompressor.class.getName());


    /**
     *  Constructor.
     */
    public CryoCompressor() {
        super(new CryoCompState());
    }
            

    /**
     *  Initializes the cryo compressor control.
     */
    @Override
    public void postInit()
    {
        limitData[CompSwConds.SWC_DISC_PRESS].delayTime = -1;
        Channel phaseSepTemp = (phaseSepTempChan != null) ? channelMap.get(phaseSepTempChan) : null;
        if (phaseSepTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "phaseSepTempChan", "not specified or not defined");
        }
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].channel = phaseSepTemp;
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].immedLimit = Double.NaN;
        Channel suctPress = (suctPressChan != null) ? channelMap.get(suctPressChan) : null;
        if (suctPress == null) {
            ErrorUtils.reportConfigError(LOG, name, "suctPressChan", "not specified or not defined");
        }
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].channel = discTemp;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].isLower = true;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].noShutoff = true;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].delayLimit = Double.NaN;
        limitData[CompSwConds.SWC_PRESS_DIFF].channel = discPress;
        limitData[CompSwConds.SWC_PRESS_DIFF].channel2 = suctPress;
        limitData[CompSwConds.SWC_PRESS_DIFF].noShutoff = true;
        limitData[CompSwConds.SWC_PRESS_DIFF].delayLimit = Double.NaN;
        switchDevices[CompSwitches.SW_HEATER] = plutoDevc;
        switchChannels[CompSwitches.SW_HEATER] = CompPlutoDevice.SW_HEATER;
        switchDevices[CompSwitches.SW_ORIFICE_VALVE] = maq20Devc;
        switchChannels[CompSwitches.SW_ORIFICE_VALVE] = CompMaq20Device.SW_ORIFICE_VALVE;
        switchDevices[CompSwitches.SW_COOLANT_VALVE] = maq20Devc;
        switchChannels[CompSwitches.SW_COOLANT_VALVE] = CompMaq20Device.SW_COOLANT_VALVE;
        switchDevices[CompSwitches.SW_BYPASS_VALVE] = maq20Devc;
        switchChannels[CompSwitches.SW_BYPASS_VALVE] = CompMaq20Device.SW_BYPASS_VALVE;
        switchDevices[CompSwitches.SW_SURGE_HEATER] = maq20Devc;
        switchChannels[CompSwitches.SW_SURGE_HEATER] = CompMaq20Device.SW_SURGE_HEATER;
        super.postInit();
    }


    /**
     *  Sets the phase separator temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayLimit(double value)
    {
        phaseSepTempDelayLimit = value;
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].delayLimit = value;
    }


    /**
     *  Sets the phase separator temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPhaseSepTempDelayTime(int value)
    {
        phaseSepTempDelayTime = value;
        limitData[CompSwConds.SWC_PHASE_SEP_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the discharge temperature low immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempLowImmedLimit(double value)
    {
        discTempLowImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP_LOW].immedLimit = value;
    }


    /**
     *  Sets the discharge pressure difference immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setPressDiffImmedLimit(double value)
    {
        pressDiffImmedLimit = value;
        limitData[CompSwConds.SWC_PRESS_DIFF].immedLimit = value;
    }


    /**
     *  Updates the compressor state.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any value changed
     */
    @Override
    protected synchronized boolean updateState()
    {
        boolean changed = super.updateState();
        controlValves(state.getCompressorState() == CompressorState.RUNNING, stateChanged);
        controlFan(state.getCompressorState() == CompressorState.RUNNING, stateChanged);
        controlHeater();
        return changed;
    }


    /**
     *  Controls the cryo valves.
     * 
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @param  compOn   Whether compressor is on
     *  @param  changed  Whether compressor state has changed
     */
    private void controlValves(boolean compOn, boolean changed)
    {
        try {
            if (compOn) {
                double press = limitData[CompSwConds.SWC_DISC_PRESS].channel.getValue();
                if (press > orificeOffPress) {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, false);
                }
                else if (press < orificeOnPress) {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, true);
                }
            }
            if (changed) {
                if (compOn) {
                    setSwitch(CompSwitches.SW_BYPASS_VALVE, false);
                }
                else {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, false);
                }
                setSwitch(CompSwitches.SW_COOLANT_VALVE, !compOn);
            }
        }
        catch (RefrigException e) {
            LOG.log(Level.SEVERE, "Error operating {0} compressor valves: {1}", new Object[]{name, e});
        }
    }


    /**
     *  Controls the cryo heater.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     */
    private void controlHeater()
    {
        double temp = discTemp.getValue();
        if (temp > heaterTempLimit && state.getSwitchState(CompSwitches.SW_HEATER) == SwitchState.ON) {
            LOG.log(Level.WARNING, "Turning {0} compressor heater off - discharge temperature ({1}) > {2}",
                    new Object[]{name, temp, heaterTempLimit});
            try {
                setSwitch(CompSwitches.SW_HEATER, false);
            }
            catch (RefrigException e) {
                LOG.log(Level.SEVERE, "Error operating {0} compressor heater: {1}", new Object[]{name, e});
            }
        }
    }


    /**
     *  Controls the cryo fan.
     *
     *  @param  on  Whether compressor is on
     */
    private void controlFan(boolean compOn, boolean changed)
    {
        if (changed && pwmControl != null) {
            if (compOn) {
                pwmControl.enable();
            }
            else {
                pwmControl.disable();
            }
        }
    }


}
