package org.lsst.ccs.subsystem.refrig;

import java.io.IOException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.monitor.Alarm;
import org.lsst.ccs.subsystem.monitor.Line;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration subscale camera modular subsystem.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Subscale extends Module implements Monitor.AlarmHandler {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static int
        EVENT_ID_MAIN_POWER = 0;
    private final static double
        MAX_CURRENT = 2,
        MIN_VOLTAGE = 10;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    Monitor mon;
    String configName = "test";
    boolean coldStart, running, loadAlarm;
    int state;
    double loadPower, loadVolts, loadAmps = MAX_CURRENT, loadOhms;
    Line mainPowerLine, loadPowerLine;
    PowerDevice pwrDevc;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public Subscale(String name, int tickMillis, String configName)
    {
        super(name, tickMillis);
        this.configName = configName;
        String cold = System.getProperty("lsst.ccs.refrig.coldstart", "");
        coldStart = cold.equals("true");
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem.
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize all configuration data
        */
        mon = new Monitor(this, this, log);
        mon.initConfiguration();
        mainPowerLine = mon.getLine("MainPower");
        if (mainPowerLine == null) {
            log.error("Compressor power control line (MainPower) not defined");
        }
        loadPowerLine = mon.getLine("LoadPower");
        if (loadPowerLine == null) {
            log.error("Heater power control line (LoadPower) not defined");
        }
        pwrDevc = (PowerDevice)mon.getDevice("Power");
        if (pwrDevc == null) {
            log.error("Heater power device (Power) not defined");
        }

        /*
        **  Initialize the hardware
        */
        if (mainPowerLine != null) {
            mainPowerLine.setWarm(!coldStart);
        }
        mon.initSensors();
        if (mainPowerLine != null) {
            mainPowerLine.setWarm(false);
        }

        /*
        **  Set output lines
        */
        setMainPowerEnable(isMainPowerOn() ? 1 : 0);

        /*
        **  Start the monitoring
        */
        mon.start();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic trending data broadcast.
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
//            getSubsystem().switchToEngineeringMode();

            System.out.println("Refrigeration subscale camera test started");
            publishState();
            mon.publishState();    // For any GUIs
            mon.publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Broadcast the trending data
        */
        mon.publishData();
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period.
    **
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION, description="Sets the tick interval")
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Handles alarm events.
    **
    ***************************************************************************
    */
    @Override
    public void processAlarm(int event, int parm)
    {
        switch (event) {

        case Alarm.EVENT_TRIP:
            if (parm == EVENT_ID_MAIN_POWER) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                    state |= RefrigState.MAIN_TRIPPED_STATE;
                    setMainPowerEnable(0);
                    publishState();
               }
            }
            break;
            
        case Alarm.EVENT_RESET:
            if (parm == EVENT_ID_MAIN_POWER) {
                if ((state & RefrigState.MAIN_TRIPPED_STATE) != 0) {
                    state &= ~RefrigState.MAIN_TRIPPED_STATE;
                    publishState();
               }
            }
            break;

        default:

        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the main power enabled state on or off.
    **
    **  @param  value    The enabled state value to set: 0 = off, ~0 = on.
    **
    **  @param  publish  If true, publish the new state on the status bus.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the main power enabled state")
    public void setMainPowerEnable(int value)
    {
        if (value != 0) {
            if ((state & RefrigState.MAIN_TRIPPED_STATE) == 0) {
                state |= RefrigState.MAIN_POWER_STATE;
            }
        }
        else {
            state &= ~RefrigState.MAIN_POWER_STATE;
        }
        setMainPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power enabled state on or off.
    **
    **  @param  value  The load power enabled state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load power enabled state")
    public void setLoadPowerEnable(int value)
    {
        if (value != 0) {
            state |= RefrigState.LOAD_POWER_STATE;
        }
        else {
            state &= ~RefrigState.LOAD_POWER_STATE;
        }
        setLoadPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Sets the load power set point value.
    **
    **  @param  value  The load power set point.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Sets the load power set point")
    public void setLoadPower(double value)
    {
        loadPower = value;
        setLoadPower();
        publishState();
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data.
    **
    ***************************************************************************
    */
    @Command(type=CommandType.ACTION,
             description="Saves the current configuration")
    public void saveConfiguration() throws IOException
    {
        register(configName, "");
        mon.clearLimitChanges();
        mon.publishState();     // Must always do this
    }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the refrigeration module.
    **
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    @Command(type=CommandType.QUERY,
             description="Gets the full refrigeration state")
    public RefrigFullState getFullState()
    {
        RefrigState refgState = new RefrigState(state, getTickMillis(),
                                                loadPower);
        MonitorFullState monState = mon.getFullState();
        return new RefrigFullState(refgState, monState);
    }    


   /**
    ***************************************************************************
    **
    **  Gets the operating state word.
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    int getState()
    {
        return state;
    }


   /**
    ***************************************************************************
    **
    **  Publishes the state of the refrigeration module.
    **
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    void publishState()
    {
        if (running) {
            RefrigState refgState = new RefrigState(state, getTickMillis(),
                                                    loadPower);
            sendToStatus(new RefrigStateStatus(refgState));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Sets the main power on or off, according to its state.
    **
    ***************************************************************************
    */
    void setMainPower()
    {
        boolean on = (state & RefrigState.MAIN_POWER_STATE) != 0;
        if (mainPowerLine != null) {
            mainPowerLine.set(on);
        }
        if (loadPowerLine != null) {
            loadPowerLine.set(on);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the load power.
    **
    ***************************************************************************
    */
    void setLoadPower()
    {
        if (pwrDevc == null || !pwrDevc.isOnline()) return;

        boolean setOn = (state & RefrigState.LOAD_POWER_STATE) != 0;
        boolean pwrOn = pwrDevc.getOutput(0);
        if (!setOn) {
            if (pwrOn) {
                pwrDevc.setOutput(0, false);
            }
        }
        else {
            pwrDevc.setCurrent(0, loadAmps);
            if (loadOhms == 0) {
                pwrDevc.setVoltage(0, MIN_VOLTAGE);
                pwrDevc.setOutput(0, true);
                double amps = pwrDevc.readCurrent(0);
                loadOhms = (amps <= 0) ? 0 : pwrDevc.readVoltage(0) / amps;
            }
            loadVolts = Math.sqrt(loadPower * loadOhms);
            pwrDevc.setVoltage(0, loadVolts);
            pwrDevc.setOutput(0, true);
            for (int j = 0; j < 2; j++) {
                if (loadVolts == 0) return;
                double amps = pwrDevc.readCurrent(0);
                loadOhms = (amps <= 0) ? 0 : pwrDevc.readVoltage(0) / amps;
                loadVolts = Math.sqrt(loadPower * loadOhms);
                pwrDevc.setVoltage(0, loadVolts);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the hardware main power state
    **
    ***************************************************************************
    */
    boolean isMainPowerOn()
    {
        return mainPowerLine != null && mainPowerLine.isSet();
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state
    **
    ***************************************************************************
    */
    boolean isLoadPowerOn()
    {
        return loadPowerLine != null && loadPowerLine.isSet();
    }
    
}
