package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.sorensen.Dcs;
import org.lsst.ccs.subsystem.monitor.Channel;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;

/**
 ******************************************************************************
 **
 **  Handles a Sorensen DCS series power supply.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class DcsDevice extends Device implements PowerDevice {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    final static int
        CHAN_POWER   = 0,
        CHAN_VOLTAGE = 1,
        CHAN_CURRENT = 2,
        N_HW_CHANS   = 3;

   /**
    ***************************************************************************
    **
    **  Private lookup maps.
    **
    ***************************************************************************
    */
    private final static Map<String,Integer> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Dcs.CONN_TYPE_SERIAL);
        cTypeMap.put("FTDI",   Dcs.CONN_TYPE_FTDI);
    }

    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POWER", Channel.TYPE_POWER);
    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    String    connTypeS;   // DCS connection type string
    String    ident;       // Serial device name or FTDI serial number
    int       connType;    // DCS connection type
    Dcs       dcs;         // Associated DCS object
    double[]  values = new double[N_HW_CHANS];  // Read values 


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public DcsDevice(String cType, String id)
    {
        ident = id;
        connTypeS = cType;
    }


   /**
    ***************************************************************************
    **
    **  Performs configuration.
    **
    ***************************************************************************
    */
    @Override
    protected void configure(Monitor mon)
    {
        super.configure(mon);
        fullName = "Sorensen DCS module (" + ident + ")";
        Integer iConnType = cTypeMap.get(connTypeS);
        if (iConnType == null) {
            connType = -1;
            try {
                mon.reportError(getName(), "connection type", connTypeS);
            }
            catch (Exception e) {
            }
        }
        else {
            connType = iConnType;
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs full initialization.
    **
    ***************************************************************************
    */
    @Override
    protected void initialize()
    {
        if (connType < 0) return;
        try {
            if (!inited || dcs == null) {
                dcs = new Dcs();
            }
            dcs.open(connType, ident);
            setOnline(true);
            dcs.setOutput(false);
            dcs.setVoltage(0);
            String message = "Connected to " + fullName;
            if (!inited) {
                log.info(message);
            }
            else {
                log.error(message);
            }
        }
        catch (Exception e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            if (dcs != null) {
                close();
            }
        }
        finally {
            inited = true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection.
    **
    ***************************************************************************
    */
    @Override
    protected void close()
    {
        try {
            dcs.close();
        }
        catch (DriverException e) {
        }
        for (int j = 0; j < N_HW_CHANS; j++) {
            values[j] = 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel's parameters for validity.
    **
    ***************************************************************************
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= N_HW_CHANS) {
            mon.reportError(name, "HW channel", hwChan);
        }
        Integer iSubtype = 0, iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            mon.reportError(name, "type", type);
        }

        return new int[]{iType, iSubtype};
    }


   /**
    ***************************************************************************
    **
    **  Reads all channels as a group.
    **
    ***************************************************************************
    */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            values[CHAN_VOLTAGE] = dcs.readVoltage();
            values[CHAN_CURRENT] = dcs.readCurrent();
            values[CHAN_POWER] = values[CHAN_VOLTAGE] * values[CHAN_CURRENT];
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a channel.
    **
    ***************************************************************************
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return values[hwChan];
    }


   /**
    ***************************************************************************
    **
    **  Sets the output state.
    **
    ***************************************************************************
    */
    @Override
    public void setOutput(int chan, boolean value)
    {
        if (!testOnline()) return;
        try {
            dcs.setOutput(value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the output state.
    **
    ***************************************************************************
    */
    @Override
    public boolean getOutput(int chan)
    {
        if (!testOnline()) return false;
        try {
            return dcs.getOutput();
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the voltage.
    **
    ***************************************************************************
    */
    @Override
    public void setVoltage(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            dcs.setVoltage(value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage.
    **
    ***************************************************************************
    */
    @Override
    public double readVoltage(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return dcs.readVoltage();
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the current.
    **
    ***************************************************************************
    */
    @Override
    public void setCurrent(int chan, double value)
    {
        if (!testOnline()) return;
        try {
            dcs.setCurrent(value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the current.
    **
    ***************************************************************************
    */
    @Override
    public double readCurrent(int chan)
    {
        if (!testOnline()) return -1;
        try {
            return dcs.readCurrent();
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return -1;
        }
    }

}
