package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import javax.usb.UsbDisconnectedException;
import javax.usb.UsbEncodingException;
import javax.usb.UsbException;
import javax.usb.UsbNotActiveException;
import javax.usb.UsbNotClaimedException;
import javax.usb.UsbNotOpenException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.subsystem.refrig.data.RefrigState;

/**
 ******************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RefrigTestHdw extends RefrigTest {

   /**
    ***************************************************************************
    **
    **  Constants.
    **
    ***************************************************************************
    */
    private final static long
        CHECK_PERIOD = 10000;

   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    private List<Device> devcData = new ArrayList<>();
    private List<MccDevc> mccData = new ArrayList<>();
    private List<WuDevc> wuData = new ArrayList<>();
    private ChannelHdw[] chanDataHdw;
    private OutputLine mainPowerLine, loadPowerLine;
    private List<OutputLine> alarmLine = new ArrayList<>();


   /**
    ***************************************************************************
    **
    **  Inner container class for device description data.
    **
    ***************************************************************************
    */
    public static abstract class Device {

       /**
        ***********************************************************************
        **
        **  Constants.
        **
        ***********************************************************************
        */
        final static int
            TYPE_MCC     = 0,
            TYPE_WATTS   = 1;

       /**
        ***********************************************************************
        **
        **  Data fields.
        **
        ***********************************************************************
        */
        int           type;      // Device type code
        String        fullName;  // Full device name
        int           chanMask;  // Mask of channel IDs on this device
        RefrigTestHdw refg;      // Associated refrigeration object
        boolean       online;    // True if online (running successfully)


       /**
        ***********************************************************************
        **
        **  Performs configuration.
        **
        ***********************************************************************
        */
        void configure(RefrigTestHdw refg)
        {
            this.refg = refg;
        }


       /**
        ***********************************************************************
        **
        **  Performs basic initialization.
        **
        ***********************************************************************
        */
        abstract void initialize(boolean first);


       /**
        ***********************************************************************
        **
        **  Adds a channel.
        **
        ***********************************************************************
        */
        void addChannel(int id)
        {
            chanMask |= 1 << id;
        }


       /**
        ***********************************************************************
        **
        **  Checks the online state.
        **
        ***********************************************************************
        */
        abstract void checkOnline();


       /**
        ***********************************************************************
        **
        **  Sets the online state.
        **
        ***********************************************************************
        */
        void setOnline(boolean online)
        {
            this.online = online;
            refg.setOnline(chanMask, online);
            if (!online) {
                log.error("Disconnected from " + fullName);
                close();
            }
        }


       /**
        ***********************************************************************
        **
        **  Closes the connection.
        **
        ***********************************************************************
        */
        abstract void close();

    }


   /**
    ***************************************************************************
    **
    **  Inner container class for MCC device description data.
    **
    ***************************************************************************
    */
    public static class MccDevc extends Device {

       /**
        ***********************************************************************
        **
        **  Constants.
        **
        ***********************************************************************
        */
        final static int
            N_DEV_CHANS = 8;
        final static float[]
            RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

       /**
        ***********************************************************************
        **
        **  Data fields.
        **
        ***********************************************************************
        */
        String  dName;    // MCC device name
        String  serial;   // USB serial number
        int     did;      // MCC device ID
        MccUsb  mcc;      // Associated USB object
        int     tcMask;   // Mask of available thermocouple channels
        int     rtMask;   // Mask of available RT channels
        int     vMask;    // Mask of available voltage channels
        int     lineMask; // Mask of output line numbers on this device

       /**
        ***********************************************************************
        **
        **  Private lookup maps.
        **
        ***********************************************************************
        */
        private final static Map<String,Integer> didMap = new HashMap<>();
        static {
            didMap.put("TC_AI", MccUsb.USB_TC_AI_DID);
            didMap.put("TC",    MccUsb.USB_TC_DID);
            didMap.put("TEMP",  MccUsb.USB_TEMP_DID);
        }


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public MccDevc(String devName, String serialNo)
        {
            type = TYPE_MCC;
            dName = devName;
            serial = serialNo;
            fullName = "MCC " + dName + " module"
                         + (serial == null ? "" : " (" + serial + ")");
            Integer iDid = didMap.get(dName);
            if (iDid == null) {
                log.error("Invalid MCC device type (" + dName + ")");
                did = -1;
            }
            else {
                did = iDid;
                if (did == MccUsb.USB_TC_AI_DID) {
                    tcMask = 0x0f;
                    vMask = 0xf0;
                }
                else if (did == MccUsb.USB_TC_DID) {
                    tcMask = 0xff;
                }
                else if (did == MccUsb.USB_TEMP_DID) {
                    tcMask = 0xff;
                    rtMask = 0xff;
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Performs basic initialization.
        **
        ***********************************************************************
        */
        @Override
        void initialize(boolean first)
        {
            if (did < 0) return;
            try {
                if (first || mcc == null) {
                    mcc = new MccUsb();
                }
                mcc.open(did, serial, true);
                mcc.blink();
                mcc.dioConfig(0);
                if (did == MccUsb.USB_TC_AI_DID) {
                    for (int j = 0; j < N_DEV_CHANS; j++) {
                        mcc.configAlarm(j, 0, 0, 0f, 0f);
                    }
                }
                else if (did == MccUsb.USB_TEMP_DID) {
                    for (int j = 0; j < N_DEV_CHANS; j += 2) {
                        mcc.setSensorType(j, MccUsb.STP_DISABLED);
                    }
                }
                setOnline(true);
                String message = "Connected to " + fullName;
                if (first) {
                    log.info(message);
                }
                else {
                    log.error(message);
                }
            }
            catch (Exception e) {
                checkUsbException(e);
                if (first) {
                    log.error("Error connecting to " + fullName + ": " + e);
                }
                if (mcc != null) {
                    close();
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Adds an output line.
        **
        ***********************************************************************
        */
        void addLine(int id)
        {
            lineMask |= 1 << id;
        }


       /**
        ***********************************************************************
        **
        **  Checks the online state.
        **
        ***********************************************************************
        */
        @Override
        void checkOnline()
        {
            if (online) return;
            initialize(false);
            if (!online) return;
            for (int id = 0; id < RefrigState.MAX_CHANNELS; id++) {
                if ((chanMask & (1 << id)) != 0) {
                    ((ChannelHdw)refg.getChannel(id)).initSensor();
                }
            }
            if (lineMask != 0) {
                refg.setOutputLines();
            }
        }


       /**
        ***********************************************************************
        **
        **  Closes the connection.
        **
        ***********************************************************************
        */
        @Override
        void close()
        {
            try {
                mcc.close();
            }
            catch (Exception e) {
                checkUsbException(e);
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner container class for WattsUp meter description data.
    **
    ***************************************************************************
    */
    public static class WuDevc extends Device implements WattsUp.Listener {

       /**
        ***********************************************************************
        **
        **  Constants.
        **
        ***********************************************************************
        */
        final static int
            CHAN_WATTS  = 0,
            CHAN_VOLTS  = 1,
            N_DEV_CHANS = 2,
            LOG_PERIOD  = 1;

       /**
        ***********************************************************************
        **
        **  Data fields.
        **
        ***********************************************************************
        */
        String  node;        // Name of the node attached to the meter
        String  serial;      // The serial number of the meter
        WattsUp wtu;         // WattsUp hardware object
        float[] value;       // Current values: power, voltage


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public WuDevc(String nodeName, String serialNo)
        {
            type = TYPE_WATTS;
            node = nodeName;
            serial = serialNo;
            fullName = "WattsUp meter (" + (node == null ? "local" : node)
                         + (serial == null ? "" : ":" + serial) + ")";
            value = new float[N_DEV_CHANS];
        }


       /**
        ***********************************************************************
        **
        **  Performs basic initialization.
        **
        ***********************************************************************
        */
        @Override
        void initialize(boolean first)
        {
            try {
                if (first || wtu == null) {
                    wtu = new WattsUp();
                    wtu.addListener(this);
                }
                wtu.open(node, 0, serial);
                wtu.restart();
                setOnline(true);
                String message = "Connected to " + fullName;
                if (first) {
                    log.info(message);
                }
                else {
                    log.error(message);
                }
            }
            catch (Exception e) {
                if (first) {
                    log.error("Error connecting to " + fullName + ": " + e);
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Checks the online state.
        **
        ***********************************************************************
        */
        @Override
        void checkOnline()
        {
            if (online) return;
            initialize(false);
        }


       /**
        ***********************************************************************
        **
        **  Closes the connection.
        **
        **  This is a no-op
        **
        ***********************************************************************
        */
        @Override
        void close()
        {
        }


       /**
        ***********************************************************************
        **
        **  Changes the WattsUp? meter powered state.
        **
        **  Performs meter setup upon power-on.
        **
        **  @param  on  Whether the meter is powered on or not
        **
        ***********************************************************************
        */
        @Override
        public void setPowered(boolean on)
        {
            if (on) {
                try {
                    wtu.setLoggedFields((1 << WattsUp.FLD_WATTS) |
                                        (1 << WattsUp.FLD_VOLTS));
                    wtu.setExternalLogging(LOG_PERIOD);
                }
                catch (Exception e) {
                    log.error("Error configuring " + fullName + ": " + e);
                }
            }
            else {
                for (int j = 0; j < N_DEV_CHANS; j++) {
                    value[j] = 0F;
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Changes the WattsUp? meter open state.
        **
        ***********************************************************************
        */
        @Override
        public void setClosed()
        {
            setOnline(false);
            log.error("Disconnected from " + fullName);
        }


       /**
        ***********************************************************************
        **
        **  Receives data periodically from the WattsUp? meter.
        **
        **  @param  data  The array of data from the meter.
        **
        ***********************************************************************
        */
        @Override
        public void processData(float[] data)
        {
            value[CHAN_WATTS] = data[WattsUp.FLD_WATTS];
            value[CHAN_VOLTS] = data[WattsUp.FLD_VOLTS];
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner container class for channel hardware data.
    **
    ***************************************************************************
    */
    public static class ChannelHdw extends Channel {

       /**
        ***********************************************************************
        **
        **  Constants.
        **
        ***********************************************************************
        */
        private final static int
            TYPE_UNKNOWN = -1,
            TYPE_TEMP    = 0,
            TYPE_VOLTS   = 1,
            TYPE_POWER   = 2,
            TYPE_SWITCH  = 3,
            TYPE_DIFF    = 4;

       /**
        ***********************************************************************
        **
        **  Data fields.
        **
        ***********************************************************************
        */
        String  typeS;       // Channel type string (TEMP, VOLTS, POWER...)
        String  subtypeS;    // Channel subtype string (extra config info)
        int     chan;        // Hardware channel for MCC channel
        int     alarm;       // Hardware alarm number
        float   offset;      // Value offset
        float   scale;       // Value scale
        int     type;        // Encoded channel type
        int     subtype;     // Encoded channel subtype
        Device  device;      // Associated device
        int     devChan;     // Channel within device

       /**
        ***********************************************************************
        **
        **  Private lookup maps.
        **
        ***********************************************************************
        */
        private final static Map<String,Integer> typeMap = new HashMap<>();
        static {
            typeMap.put("TEMP",   TYPE_TEMP);
            typeMap.put("VOLTS",  TYPE_VOLTS);
            typeMap.put("POWER",  TYPE_POWER);
            typeMap.put("SWITCH", TYPE_SWITCH);
            typeMap.put("DIFF",   TYPE_DIFF);
        }

        private final static Map<String,Integer> tSbTypMap = new HashMap<>();
        static {
            tSbTypMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
            tSbTypMap.put("RTD", MccUsb.STP_RTD);
        }

        private final static Map<String,Integer> tcSbTypMap = new HashMap<>();
        static {
            tcSbTypMap.put("J", MccUsb.TC_TYPE_J);
            tcSbTypMap.put("K", MccUsb.TC_TYPE_K);
            tcSbTypMap.put("T", MccUsb.TC_TYPE_T);
            tcSbTypMap.put("E", MccUsb.TC_TYPE_E);
            tcSbTypMap.put("R", MccUsb.TC_TYPE_R);
            tcSbTypMap.put("S", MccUsb.TC_TYPE_S);
            tcSbTypMap.put("B", MccUsb.TC_TYPE_B);
            tcSbTypMap.put("N", MccUsb.TC_TYPE_N);
        }

        private final static Map<String,Integer> rtSbTypMap = new HashMap<>();
        static {
            rtSbTypMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
            rtSbTypMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
            rtSbTypMap.put("3WIRE",  MccUsb.TCT_3WIRE);
            rtSbTypMap.put("4WIRE",  MccUsb.TCT_4WIRE);
        }

        private final static Map<String,Integer> vSbTypMap = new HashMap<>();
        static {
            vSbTypMap.put("10V",   MccUsb.RANGE_10V);
            vSbTypMap.put("5V",    MccUsb.RANGE_5V);
            vSbTypMap.put("2.5V",  MccUsb.RANGE_2_5V);
            vSbTypMap.put("1.25V", MccUsb.RANGE_1_25V);
        }


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public ChannelHdw(String desc, String units,
                          String loCheck, double limitLo, double deadbandLo,
                          String hiCheck, double limitHi, double deadbandHi,
                          String type, String subtype,
                          int chan, int alarm, double offset, double scale)
        {
            super(desc, units, loCheck, limitLo, deadbandLo, hiCheck, limitHi,
                  deadbandHi);
            this.typeS    = type;
            this.subtypeS = subtype;
            this.chan     = chan;
            this.alarm    = alarm;
            this.offset   = (float)offset;
            this.scale    = (float)scale;
        }


       /**
        ***********************************************************************
        **
        **  Configures direct channel data.
        **
        ***********************************************************************
        */
        void configure(RefrigTestHdw refg, int id)
        {
            super.configure(refg, id);
            int iType = TYPE_UNKNOWN, iSubtype = -1, iChan = -1,
                iAlarm = -1;
            try {
                Integer iTyp = typeMap.get(typeS.toUpperCase());
                if (iTyp == null) {
                    reportError(getName(), "type", typeS);
                }
                iType = iTyp;

                String sbtype = subtypeS.toUpperCase();
                if (iType == TYPE_TEMP) {
                    String[] sbtypes = sbtype.split(":", -1);
                    if (sbtypes.length < 2) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    Integer iStp = tSbTypMap.get(sbtypes[0]);
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype = iStp;
                    if (iStp == MccUsb.STP_RTD) {
                        iStp = rtSbTypMap.get(sbtypes[1]);
                    }
                    else {
                        iStp = tcSbTypMap.get(sbtypes[1]);
                    }
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype |= (iStp << 8);
                }
                else if (iType == TYPE_VOLTS) {
                    Integer iStp = vSbTypMap.get(sbtype);
                    if (iStp == null) {
                        reportError(getName(), "subtype", subtypeS);
                    }
                    iSubtype = iStp;
                }

                boolean chanOk = true;
                if (iType == TYPE_TEMP || iType == TYPE_VOLTS
                      || iType == TYPE_SWITCH) {
                    device = refg.getDevice(Device.TYPE_MCC, chan);
                    devChan = refg.getDevChan(Device.TYPE_MCC, chan);
                    if (device != null) {
                        MccDevc mccDev = (MccDevc)device;
                        if (iType == TYPE_TEMP) {
                            int mask = mccDev.tcMask;
                            if ((iSubtype & 0xff) == MccUsb.STP_RTD) {
                                mask = mccDev.rtMask;
                            }
                            if (((1 << devChan) & mask) == 0) {
                                chanOk = false;
                            }
                        }
                        else if (iType == TYPE_VOLTS) {
                            if (((1 << devChan) & mccDev.vMask) == 0) {
                                chanOk = false;
                            }
                        }
                    }
                    else {
                        chanOk = false;
                    }
                }
                else if (iType == TYPE_POWER) {
                    device = refg.getDevice(Device.TYPE_WATTS, chan);
                    devChan = refg.getDevChan(Device.TYPE_WATTS, chan);
                    if (device == null) {
                        chanOk = false;
                    }
                }
                if (!chanOk) {
                    reportError(getName(), "hw channel number", chan);
                }
                if (device != null) {
                    device.addChannel(id);
                }
                iChan = chan;

                if (!refg.checkAlarm(alarm)) {
                    reportError(getName(), "alarm number", alarm);
                }
                iAlarm = alarm;
            }
            catch (Exception e) {
            }

            type     = iType;
            subtype  = iSubtype;
            chan     = iChan;
            alarm    = iAlarm;
        }


       /**
        ***********************************************************************
        **
        **  Configures derived channel data.
        **
        ***********************************************************************
        */
        void configDerived()
        {
            if (type != TYPE_DIFF) return;
            String[] names = subtypeS.split(":", -1);
            boolean okay = false;
            if (names.length == 2) {
                RefrigTestHdw refgHdw = (RefrigTestHdw)refg;
                int id0 = refgHdw.findChannelId(names[0]);
                int id1 = refgHdw.findChannelId(names[1]);
                if (id0 >= 0 && id1 >= 0) {
                    subtype = (id0 << 16) | id1;
                    okay = true;
                    refg.setOnline(1 << id, true);
                }
            }
            if (!okay) {
                log.error("Invalid subtype (" + subtypeS + ") for "
                            + getName());
                type = TYPE_UNKNOWN;
            }
        }


       /**
        ***********************************************************************
        **
        **  Initializes the hardware sensor.
        **
        ***********************************************************************
        */
        void initSensor()
        {
            if (device == null || device.type != Device.TYPE_MCC
                  || !device.online) return;
            MccUsb mcc = ((MccDevc)device).mcc;
            try {
                if (type == TYPE_TEMP) {
                    int sType = subtype & 0xff, tType = subtype >> 8;
                    mcc.setSensorType(devChan, sType);
                    if (sType == MccUsb.STP_RTD) {
                        mcc.setThermConnType(devChan, tType);
                        mcc.setExcitation(devChan, MccUsb.CEX_210UA);
                        mcc.setGain(devChan, MccUsb.GAIN_4X);
                        for (int j = 0; j < MccDevc.RTD_COEFFS.length; j++) {
                            mcc.setCoefficient(devChan, j,
                                               MccDevc.RTD_COEFFS[j]);
                        }
                    }
                    else {
                        mcc.setTCType(devChan, tType);
                    }
                }
                else if (type == TYPE_VOLTS) {
                    mcc.setGain(devChan, subtype);
                    mcc.setVoltageConnType(devChan, MccUsb.VCT_DIFFERENTIAL);
                }
                else if (type == TYPE_SWITCH) {
                    mcc.dioConfigBit(devChan, 1);
                }
            }
            catch (Exception e) {
                checkUsbException(e);
                log.error("Error configuring " + device.fullName + ": " + e);
                device.setOnline(false);
            }
        }


       /**
        ***********************************************************************
        **
        **  Reads the hardware sensor.
        **
        ***********************************************************************
        */
        void readSensor()
        {
            if (device == null) return;

            if (device.type == Device.TYPE_WATTS) {
                value = ((WuDevc)device).value[devChan];
            }

            else if (device.type == Device.TYPE_MCC) {
                if (!device.online) return;
                MccUsb mcc = ((MccDevc)device).mcc;
                if (type == TYPE_SWITCH) {
                    try {
                        value = mcc.dioInBit(devChan);
                    }
                    catch (Exception e) {
                        checkUsbException(e);
                        log.error("Error reading DIO line: " + e);
                        device.setOnline(false);
                    }
                }
                else {
                    try {
                        value = offset + scale * mcc.adcIn(devChan, 0);
                    }
                    catch (Exception e) {
                        checkUsbException(e);
                        log.error("Error reading ADC: " + e);
                        device.setOnline(false);
                    }
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Calculates derived values.
        **
        ***********************************************************************
        */
        void calcDerived()
        {
            if (type != TYPE_DIFF) return;
            value = refg.getChannel(subtype >> 16).value
                      - refg.getChannel(subtype & 0xffff).value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner container class for output line data
    **
    ***************************************************************************
    */
    public static class OutputLine {

       /**
        ***********************************************************************
        **
        **  Constants.
        **
        ***********************************************************************
        */
        private final static int
            TYPE_UNKNOWN    = -1,
            TYPE_MAIN_POWER = 0,
            TYPE_LOAD_POWER = 1,
            TYPE_ALARM      = 2;

       /**
        ***********************************************************************
        **
        **  Data fields.
        **
        ***********************************************************************
        */
        String  typeS;       // Line type string (MAIN, LOAD, ALARM)
        int     chan;        // MCC channel number
        int     chanC;       // Complementary MCC channel number
        int     type;        // Encoded type
        MccDevc device;      // Device for this line
        int     devChan;     // Channel number within the device
        MccDevc deviceC;     // Device for the complementary line
        int     devChanC;    // Complementary channel number within the device

       /**
        ***********************************************************************
        **
        **  Private lookup maps.
        **
        ***********************************************************************
        */
        private final static Map<String,Integer> typeMap = new HashMap<>();
        static {
            typeMap.put("MAIN",  TYPE_MAIN_POWER);
            typeMap.put("LOAD",  TYPE_LOAD_POWER);
            typeMap.put("ALARM", TYPE_ALARM);
        }


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public OutputLine(String type, int chan, int chanC)
        {
            this.typeS = type;
            this.chan  = chan;
            this.chanC = chanC;
        }


       /**
        ***********************************************************************
        **
        **  Configures the line.
        **
        ***********************************************************************
        */
        void configure(RefrigTestHdw refg, int id)
        {
            Integer iType = typeMap.get(typeS);
            if (iType == null) {
                log.error("Invalid output line type: " + typeS);
                type = TYPE_UNKNOWN;
                return;
            }
            type = iType;
            if (chan >= 0) {
                device = (MccDevc)refg.getDevice(Device.TYPE_MCC, chan);
                devChan = refg.getDevChan(Device.TYPE_MCC, chan);
                if (device != null) {
                    device.addLine(id);
                }
                else {
                    log.error("Invalid output line channel: " + chan);
                }
            }
            if (chanC >= 0) {
                deviceC = (MccDevc)refg.getDevice(Device.TYPE_MCC, chanC);
                devChanC = refg.getDevChan(Device.TYPE_MCC, chanC);
                if (deviceC != null) {
                    deviceC.addLine(id);
                }
                else {
                    log.error("Invalid output line channel: " + chanC);
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets the line value.
        **
        ***********************************************************************
        */
        void set(boolean on)
        {
            int value = on ? 1 : 0;
            if (device != null && device.online) {
                try {
                    device.mcc.dioOutBit(devChan, value);
                }
                catch (Exception e) {
                    checkUsbException(e);
                    log.error("Error setting DIO line: " + e);
                    device.setOnline(false);
                }
            }
            if (deviceC != null && deviceC.online) {
                try {
                    deviceC.mcc.dioOutBit(devChanC, value ^ 1);
                }
                catch (Exception e) {
                    checkUsbException(e);
                    log.error("Error setting DIO line: " + e);
                    deviceC.setOnline(false);
                }
            }
        }


       /**
        ***********************************************************************
        **
        **  Gets the line value.
        **
        ***********************************************************************
        */
        boolean isSet()
        {
            int value = 0;
            if (device != null && device.online) {
                try {
                    value = device.mcc.dioInBit(devChan);
                }
                catch (Exception e) {
                    checkUsbException(e);
                    log.error("Error reading DIO line: " + e);
                    device.setOnline(false);
                }
            }
            return value != 0;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to check hardware devices state.
    **
    ***************************************************************************
    */
    private class CheckDevices extends TimerTask {

        @Override
        public void run()
        {
            for (Device devc : devcData) {
                devc.checkOnline();
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public RefrigTestHdw(String name, int tickMillis, String configName)
    {
        super(name, tickMillis, configName);
    }


   /**
    ***************************************************************************
    **
    **  Configures the devices and channels with description data.
    **
    ***************************************************************************
    */
    @Override
    void initConfiguration()
    {
        for (MccDevc mccDev : getChildren(MccDevc.class).values()) {
            mccData.add(mccDev);
            mccDev.configure(this);
            devcData.add(mccDev);
        }

        for (WuDevc wuDev : getChildren(WuDevc.class).values()) {
            wuData.add(wuDev);
            wuDev.configure(this);
            devcData.add(wuDev);
        }

        int id = 0;
        for (OutputLine line : getChildren(OutputLine.class).values()) {
            line.configure(this, id++);
            if (line.type == OutputLine.TYPE_MAIN_POWER) {
                mainPowerLine = line;
            }
            else if (line.type == OutputLine.TYPE_LOAD_POWER) {
                loadPowerLine = line;
            }
            else if (line.type == OutputLine.TYPE_ALARM) {
                alarmLine.add(line);
            }
        }

        Map<String, ChannelHdw> chanMap = getChildren(ChannelHdw.class);
        chanData = new ChannelHdw[chanMap.size()];
        chanDataHdw = (ChannelHdw[])chanData;
        id = 0;
        for (ChannelHdw ch : chanMap.values()) {
            ch.configure(this, id);
            chanDataHdw[id++] = ch;
        }
        for (ChannelHdw ch : chanDataHdw) {
            ch.configDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors.
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Perform basic initialization of all devices
        */
        for (Device devc : devcData) {
            devc.initialize(true);
        }

        /*
        **  Initialize all the hardware channels
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.initSensor();
        }

        /*
        **  Start the timer task which checks the device state periodically
        */
        (new Timer(true)).schedule(new CheckDevices(), 0L, CHECK_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data.
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        /*
        **  Read all the hardware channels
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.readSensor();
        }

        /*
        **  Compute derived values
        */
        for (ChannelHdw ch : chanDataHdw) {
            ch.calcDerived();
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns the main power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setMainPower(boolean on)
    {
        if (mainPowerLine == null) return;
        mainPowerLine.set(on);
    }
            

   /**
    ***************************************************************************
    **
    **  Turns the load power on or off.
    **
    ***************************************************************************
    */
    @Override
    void setLoadPower(boolean on)
    {
        if (loadPowerLine == null) return;
        loadPowerLine.set(on);
    }


   /**
    ***************************************************************************
    **
    **  Gets the hardware main power state
    **
    ***************************************************************************
    */
    @Override
    boolean isMainPowerOn()
    {
        boolean on = false;
        if (mainPowerLine != null) {
            on = mainPowerLine.isSet();
        }

        return on;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the hardware load power state
    **
    ***************************************************************************
    */
    @Override
    boolean isLoadPowerOn()
    {
        boolean on = false;
        if (loadPowerLine != null) {
            on = loadPowerLine.isSet();
        }

        return on;
    }
            

   /**
    ***************************************************************************
    **
    **  Sets the front-panel display lights.
    **
    ***************************************************************************
    */
    @Override
    void setDisplay(int mask, int state)
    {
        for (ChannelHdw ch : chanDataHdw) {
            if (ch.alarm < 0 || (mask & (1 << ch.id)) == 0) continue;
            alarmLine.get(ch.alarm).set(((state >> ch.id) & 1) == 0);
        }
    }


   /**
    ***************************************************************************
    **
    **  Turns an alarm on or off.
    **
    ***************************************************************************
    */
    void setAlarm(int alarm, boolean on)
    {
        if (alarm < 0 || alarm >= alarmLine.size()) return;
        alarmLine.get(alarm).set(on);
    }


   /**
    ***************************************************************************
    **
    **  Checks an alarm number for validity
    **
    ***************************************************************************
    */
    boolean checkAlarm(int alarm)
    {
        return alarm < alarmLine.size();
    }


   /**
    ***************************************************************************
    **
    **  Gets the device for a given channel type and number.
    **
    ***************************************************************************
    */
    private Device getDevice(int type, int chan)
    {
        Device devc = null;
        if (type == Device.TYPE_MCC) {
            if (chan >= 0 && chan < MccDevc.N_DEV_CHANS * mccData.size()) {
                devc = mccData.get(chan / MccDevc.N_DEV_CHANS);
            }
        }
        else if (type == Device.TYPE_WATTS) {
            if (chan >= 0 && chan < WuDevc.N_DEV_CHANS * wuData.size()) {
                devc = wuData.get(chan / WuDevc.N_DEV_CHANS);
            }
        }

        return devc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the device channel for a given channel type and number.
    **
    ***************************************************************************
    */
    private int getDevChan(int type, int chan)
    {
        int devChan = -1;
        if (type == Device.TYPE_MCC) {
            devChan = chan % MccDevc.N_DEV_CHANS;
        }
        else if (type == Device.TYPE_WATTS) {
            devChan = chan % WuDevc.N_DEV_CHANS;
        }

        return devChan;
    }


   /**
    ***************************************************************************
    **
    **  Checks whether an exception is USB-related
    **
    ***************************************************************************
    */
    private static void checkUsbException(Exception e)
    {
        if (e instanceof UsbException
            || e instanceof UsbDisconnectedException
            || e instanceof UsbEncodingException
            || e instanceof UsbNotActiveException
            || e instanceof UsbNotOpenException
            || e instanceof UsbNotClaimedException) return;
        throw (RuntimeException)e;
    }

}
