package org.lsst.ccs.subsystem.refrig.constants;

import java.util.HashMap;
import java.util.Map;

/**
 *  Error-word and Warning-word bit names (from Chiller manual)
 */

public class ChillerErrorTables {

    public enum Error1 {
	
        E1_GPIB_RX_OVERFLOW            ("0x00000001"),
        E1_FAIL_SAFE_ENGAGED           ("0x00000002"),
        E1_RTD1_NO_PROBE_CAL           ("0x00000004"),
        E1_LINE_FREQ_NOT_DETECTED      ("0x00000008"),
        E1_CODE_BASE_CHANGED           ("0x00000010"),
        E1_LOW_FLUID                   ("0x00000020"),
        E1_ANALOG_INPUT                ("0x00000040"),
        E1_RTD1                        ("0x00000080"),
        E1_RTD2                        ("0x00000100"),
        E1_TC                          ("0x00000200"),
        E1_CJ                          ("0x00000400"),
        E1_FRONT_PANEL_MISSING         ("0x00000800"),
        E1_ETH_USB_MISSING             ("0x00001000"),
        E1_TRANSLATOR_MISSING          ("0x00002000"),
        E1_REMOTE_DIGITAL_TEMP_MISSING ("0x00004000"),
        E1_FACTORY_NVM_NOT_SET         ("0x00008000"),
        E1_TEMP_RUN_AWAY               ("0x00010000"),
        E1_TC_CAL_BAD                  ("0x00020000"),
        E1_VFD_FAULT                   ("0x00040000"),
        E1_HIGH_PRESSURE_FAULT         ("0x00080000"),
        E1_CONTROLLER_TEMP_LOW         ("0x00100000"),
        E1_CONTROLLER_TEMP_HIGH        ("0x00200000"),
        E1_SERIAL_RX_OVERFLOW          ("0x00400000"),
        E1_REMOTE_SERIAL_COM           ("0x00800000"),
        E1_PROGRAM_ILLEGAL_STEP        ("0x01000000"),
        E1_RTD2_NO_PROBE_CAL           ("0x02000000"),
        E1_TANK_TOO_FULL               ("0x04000000"),
        E1_TCP_RX_OVERFLOW             ("0x08000000"),
        E1_NVM_NOT_SAVED_BAD_PARAM     ("0x10000000"),
        E1_BIT29_FutureUse             ("0x20000000"),
        E1_AD_NO_DATA                  ("0x40000000"),
        E1_PID_PARAMS_BAD              ("0x80000000");

	private String pattern;

	Error1(String pattern) {
	    this.pattern = pattern;
	}

	public String getPattern() {return pattern;}
    }

    public enum Error2 {

        E2_SERIAL_TX_OVERFLOW         ("0x00000001"),
        E2_BIT01_FutureUse            ("0x00000002"),
        E2_LOW_FLOW_RATE_FROM_SWITCH  ("0x00000004"), 
        E2_BIT03_FutureUse            ("0x00000008"),
        E2_AC_PHASE_FAULT             ("0x00000010"),
        E2_EMERGENCY_SWITCH_DETECTED  ("0x00000020"), 
        E2_BIT06_FutureUse            ("0x00000040"),
        E2_COMPRESSOR1_FAIL           ("0x00000080"),
        E2_COMPRESSOR2_FAIL           ("0x00000100"),
        E2_BIT09_FutureUse            ("0x00000200"),
        E2_LOW_FLOW_RATE_FROM_METER   ("0x00000400"),  
        E2_BIT11_FutureUse            ("0x00000800"),
        E2_BAD_EEPROM                 ("0x00001000");

	private String pattern;

	Error2(String pattern) {
	    this.pattern = pattern;
	}

	public String getPattern() {return pattern;}
    }

    public enum Warning1 {
    
        W_RESET_FROM_BROWN_OUT             ("0x00000001"),
        W_RESET_FROM_WDT                   ("0x00000002"),
        W_TOD_BAD                          ("0x00000004"),
        W_FIRMWARE_NOT_UPDATED_FROM_VB_APP ("0x00000008"),
        W_RAMP_TIMEOUT                     ("0x00000010"),
        W_SD_CARD_INIT                     ("0x00000020"),
        W_SD_CARD_MISSING                  ("0x00000040"),
        W_NVM_SET_TO_DEFAULTS              ("0x00000080"),
        W_LOW_N2                           ("0x00000100"),
        W_LOW_FLUID                        ("0x00000200"),
        W_SD_CARD_WRITE_PROTECTED          ("0x00000400"),
        W_SD_CARD_WRITE_ERROR              ("0x00000800"),
        W_USB_STICK_MISSING                ("0x00001000"),
        W_USB_STICK_WRITE_PROTECTED        ("0x00002000"),
        W_USB_STICK_INIT                   ("0x00004000"),
        W_USB_STICK_WRITE_ERROR            ("0x00008000"),
        W_NOT_RELEASED_FIRMWARE            ("0x00010000"),
        W_DRIP_TRAY_FULL                   ("0x00020000"),
        W_TOD_NOT_SET_CORRECTLY            ("0x00040000"),
        W_USB_STICK_NOT_ENOUGH_MEMORY      ("0x00080000"),
        W_SYS_COMM_CHECKSUM                ("0x00100000"),
        W_SYS_COMM_OVERRUN                 ("0x00200000"),
        W_SYS_COMM                         ("0x00400000"),
        W_SYS_DATA                         ("0x00800000"),
        W_TWO_CONSECUTIVE_WD_RESETS        ("0x01000000"),
        W_CONTROLLER_TEMP_HIGH             ("0x02000000"),
        W_CONTROLLER_TEMP_LOW              ("0x04000000"),
        W_SOFTWARE_REVISIONS_MISMATCHED    ("0x08000000"),
        W_REMOTE_SERIAL_COM                ("0x10000000"),
        W_LOW_PRESSURE                     ("0x20000000"),
        W_USER_NVM_LIFE                    ("0x40000000"),
        W_LOW_FLOW_RATE_FROM_SWITCH        ("0x80000000");

	private String pattern;

	Warning1(String pattern) {
	    this.pattern = pattern;
	}

	public String getPattern() {return pattern;}
    }
	
    public enum Warning2 {
	
	W2_CANNOT_REACH_TANK_PRESSURE  ("0x00000001");

	private String pattern;

	Warning2(String pattern) {
	    this.pattern = pattern;
	}

	public String getPattern() {return pattern;}
    }

   /**
    *  Return a Map from bit number to error name
    */
    
    public static Map<Integer, String> getError1Map() {
	Map<Integer, String> map = new HashMap<>(32);
	for (Error1 bit : Error1.values()) {
	    map.put(bit.ordinal(), bit.toString());
	}
	return map;
    }

    public static Map<Integer, String> getError2Map() {
	Map<Integer, String> map = new HashMap<>(32);
	for (Error2 bit : Error2.values()) {
	    map.put(bit.ordinal(), bit.toString());
	}
	return map;
    }

    public static Map<Integer, String> getWarning1Map() {
	Map<Integer, String> map = new HashMap<>(32);
	for (Warning1 bit : Warning1.values()) {
	    map.put(bit.ordinal(), bit.toString());
	}
	return map;
    }

    public static Map<Integer, String> getWarning2Map() {
	Map<Integer, String> map = new HashMap<>(32);
	for (Warning2 bit : Warning2.values()) {
	    map.put(bit.ordinal(), bit.toString());
	}
	return map;
    }

   /**
    *  Return table of bit patterns and error names
    */
    
    public static String getError1Table() {
	String table = "// Error1 Bits";
	for (Error1 bit : Error1.values()) {
	    table = table.concat(String.format("\n  %s    %s", bit.getPattern(),
				               bit.toString()));
	}
	return table.concat("\n");
    }

    public static String getError2Table() {
	String table = "// Error2 Bits";
	for (Error2 bit : Error2.values()) {
	    table = table.concat(String.format("\n  %s    %s", bit.getPattern(),
				               bit.toString()));
	}
	return table.concat("\n");
    }

    public static String getWarning1Table() {
	String table = "// Warning1 Bits";
	for (Warning1 bit : Warning1.values()) {
	    table = table.concat(String.format("\n  %s    %s", bit.getPattern(),
				               bit.toString()));
	}
	return table.concat("\n");
    }

    public static String getWarning2Table() {
	String table = "// Warning2 Bits";
	for (Warning2 bit : Warning2.values()) {
	    table = table.concat(String.format("\n  %s    %s", bit.getPattern(),
				               bit.toString()));
	}
	return table.concat("\n");
    }
}
