package org.lsst.ccs.subsystem.rafts;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.drivers.reb.SequencerUtils;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo.PointerKind;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;

/**
 * Routines for accessing a REB sequencer.
 *
 * @author Owen Saxton
 */
public class SequencerProc {

    /*
     *  Data fields.
     */
    private final BaseSet bss;
    private final SequencerUtils seq;
    // A register pointing to a location which is zero
    private int zeroReg;
    private FPGA2Model fpgaModel;
    
    private static final AutoCloseableReentrantLock sequencerLock = new AutoCloseableReentrantLock();

    /**
     * Constructor.
     *
     * @param bss
     */
    public SequencerProc(BaseSet bss) {
        this.bss = bss;
        seq = new SequencerUtils(bss);
    }

    /**
     * A lock which is to be taken by focal-plane when performing an idle_clear. Other tasks
     * which want to avoid interference from idle_clear need to obtain and hold this lock during
     * the time they want to avoid idle_clear from starting. If someone other than the calling thread
     * already holds the lock, the current thread will wait for the provided timeout
     * to obtain the lock. It it cannot obtain the lock in the specified time
     * period, the method will fail.
     * 
     * @param timeout The timeout to wait for the lock to become available.
     * @param units   The TimeUnit of the specified timeout
     * @return The lock obtained
     * @throws org.lsst.ccs.subsystem.rafts.AutoCloseableReentrantLock.AutoCloseableReentrantLockException If the lock is held by another thread
     */
    public static AutoCloseableReentrantLock lockSequencer(long timeout, TimeUnit units) throws AutoCloseableReentrantLock.AutoCloseableReentrantLockException {
        return sequencerLock.doLock(timeout, units);
    }

    /**
     * A lock which is to be taken by focal-plane when performing an idle_clear. Other tasks
     * which want to avoid interference from idle_clear need to obtain and hold this lock during
     * the time they want to avoid idle_clear from starting. If someone other than the calling thread
     * already holds the lock it will fail immediately.
     * @return The lock obtained
     * @throws org.lsst.ccs.subsystem.rafts.AutoCloseableReentrantLock.AutoCloseableReentrantLockException If the lock is held by another thread
     */
    public static AutoCloseableReentrantLock lockSequencer() throws AutoCloseableReentrantLock.AutoCloseableReentrantLockException {
        return sequencerLock.doLock();
    }

    /**
     * A lock which is to be taken by focal-plane when performing an idle_clear. Other tasks
     * which want to avoid interference from idle_clear need to obtain and hold this lock during
     * the time they want to avoid idle_clear from starting. If someone other than the calling thread
     * already holds the lock this method will not fail and return a null lock.
     * @return The lock obtained or null if it could not be obtained.
     */
    public static AutoCloseableReentrantLock lockSequencerIfPossible() {
        try {
            return sequencerLock.doLock();
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException e) {
            return null;
        }
    }
    
    /**
     * A lock which is to be taken by focal-plane when performing an idle_clear. Other tasks
     * which want to avoid interference from idle_clear need to obtain and hold this lock during
     * the time they want to avoid idle_clear from starting. If someone other than the calling thread
     * already holds the lock, the current thread will wait for the provided timeout
     * to obtain the lock. It it cannot obtain the lock in the specified time
     * period, this method will not fail and return a null lock.
     * 
     * @param timeout The timeout to wait for the lock to become available.
     * @param units   The TimeUnit of the specified timeout
     * @return The lock obtained or null if it could not be obtained.
     */
    public static AutoCloseableReentrantLock lockSequencerIfPossible(long timeout, TimeUnit units) {
        try {
            return sequencerLock.doLock(timeout, units);
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException e) {
            return null;
        }
    }

    /**
     * Loads a sequencer from an input string. This routine supports loading XML
     * or assembler (.seq) format.
     *
     * @param in The input stream to read
     * @return The number of slices generated by the default main
     * @throws IOException If there is an error opening or reading the file
     * @throws REBException If there is an error communicating with the REB
     */
    int loadSequencer(InputStream in) throws IOException, REBException {
        FPGA2Model model = new FPGA2ModelBuilder().compileFile(in);
        return loadSequencer(model);
    }

    /**
     * Load the sequencer from a compiled FPGA model. Note this method computes
     * many quantities, which the other methods in this class rely on.
     *
     * @param model The model to load
     * @return The number of slices generated by the default main
     * @throws REBException If there is an error talking to the REB
     *         or sequencer is running
     */
    public int loadSequencer(FPGA2Model model) throws REBException {
        try (AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS)) {
            if (isRunning()) {
                throw new REBException("Sequencer is Enabled, can't load while running");
            }
            this.fpgaModel = model;
            seq.clearCache();
            for (int j = 0; j < Sequencer.SEQ_MAX_PROGRAM; j += 16) {
                bss.write(Sequencer.REG_SEQ_PROGRAM + j + 15, 0);
            }

            for (int[] command : model.getCommands()) {
                switch (command[0]) {

                    case FPGA2Model.CMD_LINES:
                        int lines = 0;
                        for (int j = 3; j < command.length; j++) {
                            lines |= (1 << command[j]);
                        }
                        seq.writeLines(command[1], command[2], lines);
                        break;

                    case FPGA2Model.CMD_TIME:
                        seq.writeTimes(command[1], command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGFUNC:
                        seq.writeProgExec(command[1], command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGFUNC_FP:
                        seq.writeProgExec(command[1], Sequencer.SEQ_ARG_IND_FUNC, command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGFUNC_RP:
                        seq.writeProgExec(command[1], Sequencer.SEQ_ARG_IND_COUNT, command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGFUNC_FRP:
                        seq.writeProgExec(command[1], Sequencer.SEQ_ARG_IND_FUNC | Sequencer.SEQ_ARG_IND_COUNT,
                                          command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGJUMP:
                        seq.writeProgJump(command[1], command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGJUMP_AP:
                        seq.writeProgJump(command[1], Sequencer.SEQ_ARG_IND_JUMP, command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGJUMP_RP:
                        seq.writeProgJump(command[1], Sequencer.SEQ_ARG_IND_COUNT, command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_PROGJUMP_ARP:
                        seq.writeProgJump(command[1], Sequencer.SEQ_ARG_IND_JUMP | Sequencer.SEQ_ARG_IND_COUNT,
                                          command[2], command[3]);
                        break;

                    case FPGA2Model.CMD_POINTER_FA:
                        seq.writeExecFunc(command[1], command[2]);
                        break;

                    case FPGA2Model.CMD_POINTER_FR:
                        seq.writeExecCount(command[1], command[2]);
                        break;

                    case FPGA2Model.CMD_POINTER_SA:
                        seq.writeJumpSubr(command[1], command[2]);
                        break;

                    case FPGA2Model.CMD_POINTER_SR:
                        seq.writeJumpCount(command[1], command[2]);
                        break;

                    case FPGA2Model.CMD_PROGSUBE:
                        seq.writeProgEndSubr(command[1]);
                        break;

                    case FPGA2Model.CMD_PROGEND:
                        seq.writeProgEnd(command[1]);
                        break;

                    default:
                    // Silently ignored.
                }
            }
            int nSlice = seq.getCacheSliceCount();
            int version = seq.getVersion();
            if (version == BaseSet.VERSION_3) {
                seq.writeStartAddr(0);
            } else if (version == BaseSet.VERSION_0) {
                seq.writeSliceCount(nSlice);
            } // Alternatives silently ignored
            // Try to find an address for zeroReg to point to
            zeroReg = 0;
            for (int j = 0; j < Sequencer.SEQ_MAX_PROGRAM; j += 16) {
                if (bss.read(Sequencer.REG_SEQ_PROGRAM + j + 15) == 0) {
                    zeroReg = Sequencer.REG_SEQ_PROGRAM + j + 15;
                }
            }
            // DANGER: What if we did not find any ZEROs?
            return nSlice;
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException e) {
            throw new RuntimeException("Could not lock the sequencer to assess its state.",e);
        }
    }

    /**
     * Gets the list of main programs.
     *
     * @return The list of known names of main programs, or null is there are no
     * such names.
     */
    public List<String> getMains() {
        return new ArrayList<>(fpgaModel.getMainAddresses().keySet());
    }

    /**
     * Gets the map of main programs.
     *
     * @return The name-to-address map of main programs
     */
    public Map<String, Integer> getMainMap() {
        return fpgaModel.getMainAddresses();
    }

    /**
     * Gets the map of subroutines.
     *
     * @return The name-to-address map of subroutines
     */
    public Map<String, Integer> getSubroutineMap() {
        return fpgaModel.getSubroutineAddresses();
    }

    /**
     * Gets the map of functions.
     *
     * @return The name-to-number map of functions
     */
    public Map<String, Integer> getFunctionMap() {
        return fpgaModel.getFunctionAddresses();
    }

    /**
     * Gets the map of pointer information.
     *
     * @return The name-to-type-and-address map of pointers
     */
    public Map<String, PointerInfo> getPointers() {
        return fpgaModel.getPointerMap();
    }

    /**
     * Gets the array of image parameter registers to be read
     *
     * @return The array of register addresses
     */
    public int[] getImageRegisters() {
        try {
            return getImageRegisters(SequencerData.imageRegNames, false);
        } catch (RaftException ex) {
            // Should never happen, since strict is false, but just in case
            throw new RuntimeException("Inconceivable!",ex);
        }
    }
    
    /**
     * Gets the array of image parameter register addresses to be read
     * @param names The list of parameter names
     * @param strict if <code>true</code> will throw an exception if any named
     * registers are missing, otherwise just substitute an address guaranteed 
     * to point to zero.
     * @return An array of addresses
     * @throws RaftException If strict is true, 
     * and requested names are missing or invalid.
     */
    public int[] getImageRegisters(String[] names, boolean strict) throws RaftException {
        
        int[] registers = new int[names.length];
        Map<String, PointerInfo> pointerMap = fpgaModel.getPointerMap();
        int k = 0;
        for(String name : names) {
            int addr = zeroReg;
            PointerInfo pi = pointerMap.get(name);
            if (pi != null) {
                if (pi.getKind() == PointerKind.REPEAT_FUNCTION) {
                    addr = pi.getAddress();
                } else if (pi.getKind() == PointerKind.REPEAT_SUBROUTINE) {
                    addr = pi.getAddress();
                } else if (strict) {
                    throw new RaftException("Parameter: "+name+" is of unsupported type "+pi.getKind());
                }
            } else {
                if (strict) throw new RaftException("Required parameter not defined "+name);
            }
            registers[k++] = addr;
        }
        return registers;
    }

    /**
     * Sets the sequencer start address.
     *
     * @param mainName The name of the main to start at
     * @return The number of slices produced by this main
     * @throws REBException
     * @throws RaftException
     */
    public int setStart(String mainName) throws REBException, RaftException {
        Integer addr = fpgaModel.getMainAddresses().get(mainName);
        if (addr != null) {
            seq.writeStartAddr(addr);
            return seq.getCacheSliceCount();
        } else {
            throw new RaftException("Unknown main program name: " + mainName);
        }
    }

    /**
     * Sets a parameter value.
     *
     * @param parmName The name of the parameter to set
     * @param value The value to set
     * @throws REBException
     * @throws RaftException
     */
    public void setParameter(String parmName, int value)
            throws REBException, RaftException {
        // TODO: Should this also update the pointer map?
        PointerInfo pi = fpgaModel.getPointerMap().get(parmName);
        if (pi != null) {
            switch (pi.getKind()) {
                case FUNCTION:
                    seq.writeExecFunc(pi.getOffset(), value);
                    break;
                case REPEAT_FUNCTION:
                    seq.writeExecCount(pi.getOffset(), value);
                    break;
                case SUBROUTINE:
                    seq.writeJumpSubr(pi.getOffset(), value);
                    break;
                case REPEAT_SUBROUTINE:
                    seq.writeJumpCount(pi.getOffset(), value);
                    break;
            }
            return;
        }
        throw new RaftException("Unknown parameter name: " + parmName);
    }

    /**
     * Gets a parameter value.
     *
     * @param parmName The name of the parameter to set
     * @return The value
     * @throws REBException
     * @throws RaftException
     */
    public int getParameter(String parmName) throws REBException, RaftException {
        PointerInfo pi = fpgaModel.getPointerMap().get(parmName);
        if (pi != null) {
            int value = 0;
            switch (pi.getKind()) {
                case FUNCTION:
                    value = seq.readExecFunc(pi.getOffset());
                    break;
                case REPEAT_FUNCTION:
                    value = seq.readExecCount(pi.getOffset());
                    break;
                case SUBROUTINE:
                    value = seq.readJumpSubr(pi.getOffset());
                    break;
                case REPEAT_SUBROUTINE:
                    value = seq.readJumpCount(pi.getOffset());
                    break;
            }
            return value;
        }
        throw new RaftException("Unknown parameter name: " + parmName);
    }

    /**
     * Gets the data source.
     *
     * @return The encoded data source: CCD or pattern
     * @throws REBException
     */
    public int getDataSource() throws REBException {
        return seq.readDataSource();
    }

    /**
     * Sets the data source.
     *
     * @param value The encoded data source: CCD or pattern
     * @throws REBException
     */
    public void setDataSource(int value) throws REBException {
        seq.writeDataSource(value);
    }

    /**
     * Gets the scan mode enabled state.
     *
     * @return Whether scan mode is enabled
     * @throws REBException
     */
    public boolean isScanEnabled() throws REBException {
        return seq.isScanEnabled();
    }

    /**
     * Enables or disables scan mode.
     *
     * @param value The scan mode enabled state to set (true or false)
     * @throws REBException
     */
    public void enableScan(boolean value) throws REBException {
        seq.enableScan(value);
    }

    /**
     * Gets error address.
     *
     * @return The sequencer error address
     * @throws REBException
     */
    public int getErrorAddr() throws REBException {
        return seq.getErrorAddr();
    }

    /**
     * Resets error state.
     *
     * @throws REBException
     */
    private void resetError() throws REBException {
        seq.resetError();
    }

    /**
     * Starts the sequencer.
     *
     * @throws REBException
     */
    public void startSequencer() throws REBException {
        resetError();
        seq.enable();
    }

    /**
     * Sends a sequencer stop command.
     *
     * @throws REBException
     */
    public void sendStop() throws REBException {
        seq.sendStop();
    }

    /**
     * Checks whether the sequencer is running.
     *
     * @return Whether the sequencer is running
     * @throws REBException
     */
    public boolean isRunning() throws REBException {
        return seq.isEnabled();
    }

    /**
     * Waits for the sequencer to be done (idle)
     *
     * @param timeout The maximum time to wait (ms)
     * @throws REBException
     */
    public void waitDone(int timeout) throws REBException {
        seq.waitDone(timeout);
    }    

    public int getClockPeriodPicos() throws REBException {
        return bss.getClockPeriodPicos();
    }
}
