package org.lsst.ccs.subsystem.rafts;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.RegClient;
import org.lsst.ccs.framework.HasLifecycle;

/**
 * Routines for accessing the whole raft
 *
 * @author  Owen Saxton
 */
public class GlobalProc implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final List<REBDevice> rebDevices = new ArrayList();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ClientFactory clientFactory;

    @ConfigurationParameter(category=REBDevice.RAFTS, isFinal=true)
    private volatile String partition = "";

    @ConfigurationParameter(category=REBDevice.RAFTS, isFinal=true)
    private volatile String dfltFolder = "";

    private static final Logger LOG = Logger.getLogger(GlobalProc.class.getName());
    private GlobalClient gbl = new GlobalClient();
    private final List<SequencerProc> sequencers = new ArrayList<>();


    /**
     *  Life-cycle initialization.
     */
    @Override
    public void init()
    {
        if ( clientFactory != null ) {
            gbl.setClientFactory(clientFactory);
            for (REBDevice rebDevice : rebDevices) {
                rebDevice.setClientFactory(clientFactory);
            }
        }
        for (REBDevice rebDevice : rebDevices) {
            sequencers.add(rebDevice.getSequencer());
        }
        try {
            if (partition.isEmpty()) {
                int hdwType = RegClient.HDW_TYPE_DAQ1;
                int[] ids = new int[rebDevices.size()];
                String[] ifcNames = new String[rebDevices.size()];
                for (int j = 0; j < ids.length; j++) {
                    REBDevice reb = rebDevices.get(j);
                    ids[j] = reb.getId();
                    ifcNames[j] = reb.getIfcName();
                    hdwType = reb.getHdwType();
                }
                gbl.open(hdwType, ids, ifcNames);
            }
            else {
                gbl.open(partition);
            }
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error during global initialization: ", e);
        }
    }


    /**
     *  Gets the partition name.
     *
     *  Used by REBDevice during the init phase
     *
     *  @return  The partition name
     */
    public String getPartition()
    {
        return partition;
    }


    /**
     *  Sets the registers to be read during image acquisition.
     *
     *  @param  rebType    The encoded REB type
     *  @param  registers  The array of addresses of registers to be read
     *  @throws REBException
     */
    public void setRegisters(int rebType, int[] registers) throws REBException
    {
        gbl.setRegisterlist(rebType, registers);
    }


    /**
     *  Starts image acquisition.
     *
     *  @throws REBException
     */
    public void acquireImage() throws REBException
    {
        acquireImage("", "", 1, "");
    }


    /**
     *  Starts image acquisition.
     *
     *  @param  name  The image name
     *  @throws REBException
     */
    public void acquireImage(String name) throws REBException
    {
        acquireImage(name, "", 1, "");
    }


    /**
     *  Starts image acquisition.
     *
     *  @param  name    The image name
     *  @param  folder  The folder name, or "" or null to use the default
     *  @param  opcode  The opcode
     *  @param  annot   The image annotation
     *  @param  ids     Array of REBs to use for the acquisition; if empty, use whole partition
     *  @throws REBException
     */
    public void acquireImage(String name, String folder, int opcode, String annot, int... ids) throws REBException
    {
        gbl.acquireImage((name == null || name.isEmpty()) ? String.format("Image_%016x", System.currentTimeMillis()) : name,
                         (folder == null || folder.isEmpty()) ? dfltFolder : folder, opcode, annot, ids);
    }


    /**
     *  Starts sequencer (no image data).
     *
     *  @throws REBException
     */
    public void startSequencer() throws REBException
    {
        startSequencer(1);
    }


    /**
     *  Starts sequencer (no image data).
     *
     *  @param  opcode  The opcode
     *  @throws REBException
     */
    public void startSequencer(int opcode) throws REBException
    {
        if (gbl.getHwType() == RegClient.HDW_TYPE_DAQ4) {
            gbl.startSequencer(opcode);
        }
        else {
            for (SequencerProc seq : sequencers) {
                seq.resetError();
                seq.startSequencer();
            }
        }
    }

}
