package org.lsst.ccs.subsystem.rafts.ui;

import java.io.File;
import java.io.PrintStream;
import java.time.Duration;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;

/**
 *  Rafts system console program.
 *
 *  @author Owen Saxton
 */
public class RaftsConsole implements StatusMessageListener {

   /*
    *  Public data
    */
    public enum Format { RAW, POINTER }

   /*
    *  Private maps
    */
    private static final Map<Integer, String> ptrTypeMap = new HashMap<>();
    static {
         ptrTypeMap.put(SequencerData.PTR_TYPE_FUNC, "F");
         ptrTypeMap.put(SequencerData.PTR_TYPE_FUNC_REP, "FR");
         ptrTypeMap.put(SequencerData.PTR_TYPE_SUBR, "S");
         ptrTypeMap.put(SequencerData.PTR_TYPE_SUBR_REP, "SR");
    }

   /*
    *  Private fields
    */
    private static final PrintStream out = System.out;
    private static final int TIMEOUT = 10000;
    private final ConcurrentMessagingUtils ca;
    private String raftsSS;
    private int rebIndex = 0;


   /*.
    *  Trigger enumeration.
    */
    public enum Trigger {

        STAT, TIME, SEQ, TADC, PADC;

    }


   /*.
    *  Constructor.
    */
    public RaftsConsole() {
        raftsSS = System.getProperty("org.lsst.ccs.rafts.name", "ccs-rafts");
        (new Agent("RaftsConsole", AgentInfo.AgentType.CONSOLE)).startAgent();
        AgentMessagingLayer aml = Agent.getEnvironmentMessagingAccess();
        ca = new ConcurrentMessagingUtils(aml);
        aml.addStatusMessageListener(this);
    }


   /*.
    *  Sets the default REB.
    */
    @Command(name="setreb", description="Set the default REB")
    public void setReb(@Argument(name="index", description="The index of the REB")
                       int index) throws Exception
    {
        rebIndex = index;
        showReb();
    }


   /*.
    *  Shows the default REB.
    */
    @Command(name="showreb", description="Show the default REB")
    public void showReb() throws Exception
    {
        out.println("Default REB: " + getReb(rebIndex));
    }


   /*.
    *  Sets the target subsystem.
    */
    @Command(name="settarget", description="Set the target subsystem")
    public void setTarget(@Argument(name="subsys", description="The subsystem name")
                          String subsys) throws Exception
    {
        raftsSS = subsys;
        showTarget();
    }


   /*.
    *  Shows the target subsystem.
    */
    @Command(name="showtarget", description="Show the target subsystem")
    public void showTarget() throws Exception
    {
        out.println("Target subsystem: " + raftsSS);
    }


   /*.
    *  Sets the value of a register.
    */
    @Command(name="setregister", description="Set the value of a register")
    public void setRegister(@Argument(name="address", description="The register address")
                            int address,
                            @Argument(name="value", description="The value to set")
                            int value) throws Exception
    {
        setRegister(rebIndex, address, value);
    }


   /*.
    *  Sets the value of a register.
    */
    @Command(name="setregister", description="Set the value of a register")
    public void setRegister(@Argument(name="index", description="The index of the REB")
                            int index,
                            @Argument(name="address", description="The register address")
                            int address,
                            @Argument(name="value", description="The value to set")
                            int value) throws Exception
    {
        int[] values = {value};
        sendCommand(getReb(index), "setRegister", address, values);
    }


   /*.
    *  Shows the values of registers.
    */
    @Command(name="showregister", description="Show the values of registers")
    public void showRegister(@Argument(name="address", description="The first register address")
                            int address,
                            @Argument(name="count", description="The numberof registers to show")
                            int count) throws Exception
    {
        showRegister(rebIndex, address, count);
    }


   /*.
    *  Shows the values of registers.
    */
    @Command(name="showregister", description="Show the values of registers")
    public void showRegister(@Argument(name="index", description="The index of the REB")
                             int index,
                             @Argument(name="address", description="The first register address")
                             int address,
                             @Argument(name="count", description="The numberof registers to show")
                             int count) throws Exception
    {
        out.println(sendCommand(getReb(index), "getRegister", address, count));
    }


   /*.
    *  Shows all devices.
    */
    @Command(name="showdevices", description="Show all devices")
    public void showDevices() throws Exception
    {
        List<String> devices = (List<String>)sendCommand(null, "getDeviceNames");
        if (devices.isEmpty()) return;
        out.print("Devices:");
        devices.stream().forEach((devc) -> {
            out.format(" %s", devc);
        });
        out.println();
    }


   /*.
    *  Sets a configured value.
    */
    @Command(name="setvalue", description="Set a configured value")
    public void setValue(@Argument(name="chan", description="The qualified name of the channel")
                         String chan,
                         @Argument(name="value", description="The value to set")
                         String value) throws Exception
    {
        String[] wds = chan.split("\\.", 3);
        if (wds.length < 2) {
            throw new Exception("Channel name must be qualified with device name");
        }
        String field = wds[wds.length - 1];
        String channel = wds.length <= 2 ? wds[0] : wds[0] + "." + wds[1];
        String command = "get" + field.substring(0, 1).toUpperCase()
                           + field.substring(1);
        Object resp = sendCommand(channel, command);
        String rType = resp.getClass().getName();
        if (!rType.startsWith("[")) {
            if (rType.endsWith("Integer")) {
                sendCommand(channel, "change", field, Integer.decode(value));
            }
            else if (rType.endsWith("Double")) {
                sendCommand(channel, "change", field, Double.valueOf(value));
            }
            else {
                sendCommand(channel, "change", field, value);
            }
        }
        else {
            wds = value.split(",");
            if (rType.endsWith("I")) {
                int[] iResp = (int[])resp;
                if (wds.length > iResp.length) {
                    throw new Exception("Too many array elements supplied");
                }
                for (int j = 0; j < wds.length; j++) {
                    if (wds[j].equals("*")) continue;
                    iResp[j] = Integer.decode(wds[j]);
                }
                sendCommand(channel, "change", field, iResp);
            }
            else {
                double[] dResp = (double[])resp;
                if (wds.length > dResp.length) {
                    throw new Exception("Too many array elements supplied");
                }
                for (int j = 0; j < wds.length; j++) {
                    if (wds[j].equals("*")) continue;
                    dResp[j] = Double.valueOf(wds[j]);
                }
                sendCommand(channel, "change", field, dResp);
            }
        }
    }


   /*.
    *  Shows one or more configured values.
    */
    @Command(name="showvalue", description="Show one or more configured values")
    public void showValue(@Argument(name="chan", description="The name of the device or channel")
                          String chan) throws Exception
    {
        String[] wds = chan.split("\\.", 3);
        String field = wds.length == 1 ? "*" : wds[wds.length - 1];
        String channel = wds.length <= 2 ? wds[0] : wds[0] + "." + wds[1];
        if (field.equals("*") || field.length() == 0) {
            displayValues((Map)sendCommand(channel, "getConfigValues"));
        }
        else {
            String command = "get" + field.substring(0, 1).toUpperCase()
                               + field.substring(1);
            out.println("Value: " + getString(sendCommand(channel, command)));
        }
    }


   /*.
    *  Shows all configured values.
    */
    @Command(name="showvalue", description="Show all configured values")
    public void showValue() throws Exception
    {
        displayValues((Map)sendCommand(null, "getConfigValues"));
    }


   /*.
    *  Loads global DACs on a REB.
    */
    @Command(name="loaddac", description="Load all configured global DAC values")
    public void loadDac() throws Exception
    {
        sendCommand(getReb(rebIndex), "loadDacs");
    }


   /*.
    *  Loads bias DACs on a REB.
    */
    @Command(name="loadbias", description="Load all configured bias DAC values")
    public void loadBias() throws Exception
    {
        sendCommand(getReb(rebIndex), "loadBiasDacs");
    }


   /*.
    *  Loads ASPICs on a REB.
    */
    @Command(name="loadaspic", description="Load all configured ASPIC values")
    public void loadAspic() throws Exception
    {
        sendCommand(getReb(rebIndex), "loadAspics");
    }


   /*.
    *  Checks ASPICs on a REB.
    */
    @Command(name="checkaspic", description="Check all configured ASPIC values")
    public void checkAspic() throws Exception
    {
        List diff = (List)sendCommand(getReb(rebIndex), "checkAspics");
        out.println("No. of differences: " + diff.size());
    }


   /*.
    *  Sets the time on all REBs to the Unix system time.
    */
    @Command(name="settime", description="Set the time to the Unix system time")
    public void setTime() throws Exception
    {
        sendCommand(null, "setTime");
    }


   /*.
    *  Shows the current time on a REB.
    */
    @Command(name="showtime", description="Show the current time on an REB")
        public void showTime(@Argument(name="index", description="The index of the REB")
                             int index) throws Exception
    {
        String device = getReb(index);
        displayTime(device, (Long)sendCommand(device, "getTime"));
    }


   /*.
    *  Shows the current times on all REBs.
    */
    @Command(name="showtime", description="Show the current times on all REBs")
    public void showTime() throws Exception
    {
        List times = (List)sendCommand(null, "getTime");
        for (int j = 0; j < times.size(); j += 2) {
            displayTime((String)times.get(j), (Long)times.get(j + 1));
        }
    }


   /*.
    *  Shows a trigger time on an REB.
    */
    @Command(name="showtime", description="Show a trigger time on a REB")
    public void showTime(@Argument(name="index", description="The index of the REB")
                         int index,
                         @Argument(name="trigger", description="The name of the trigger")
                         Trigger trigger) throws Exception
    {
        displayTime("Trigger", (Long)sendCommand(getReb(index), "getTime", trigger.name()));
    }


   /*.
    *  Checks the sequencer code compiled from a local XML file.
    */
    @Command(name="checkxml", description="Check the sequencer code from a local XML file")
    public void checkXml(@Argument(name="file", description="The name of the file")
                         String file) throws Throwable
    {
        getModel(file);
    }


   /*.
    *  Dumps the sequencer code compiled from a local XML file.
    */
    @Command(name="dumpxml", description="Dump the sequencer code from a local XML file")
    public void dumpXml(@Argument(name="file", description="The name of the file")
                        String file) throws Throwable
    {
        FPGA2Model model = getModel(file);
        List<int[]> commands = model.getCommands();
        commands.stream().forEach((cmnd) -> {
            for (int j = 0; j < cmnd.length; j++) {
                String fmt = j == 0 ? "%x" : j == 1 ? " %2d" : " %4x";
                out.format(fmt, cmnd[j]);
            }
            out.println();
        });
    }


   /*.
    *  Dumps the sequencer code (address + value) compiled from a local XML file.
    */
    @Command(name="dumpxmlraw", description="Dump the sequencer code from a local XML file")
    public void dumpXmlRaw(@Argument(name="file", description="The name of the file")
                           String file) throws Throwable
    {
        FPGA2Model model = getModel(file);
        List<FPGA2Model.AddressAndValue> data = model.getMemoryMap();
        data.stream().forEach((item) -> {
            out.format("0x%06x 0x%08x\n", item.getAddress(), item.getValue());
        });
    }


   /*.
    *  Loads the sequencer from a remote file.
    */
    @Command(name="loadsequencer", description="Load the sequencer from a remote file")
    public void loadSequencer(@Argument(name="file", description="The name of the file")
                              String file) throws Exception
    {
        List<Integer> slices = (List<Integer>)sendCommand(null, "loadSequencer", file);
        out.print("Slice counts:");
        slices.stream().forEach((nSlice) -> {
            out.format(" %s", nSlice);
        });
        out.println();
    }


   /*.
    *  Shows the sequencer main programs.
    */
    @Command(name="showmains", description="Show the sequencer main programs")
    public void showMains() throws Exception
    {
        List<Map<String, Integer>> maps;
        maps = (List<Map<String, Integer>>)sendCommand(null, "getMainMap");
        displayMaps("Mains", maps, Format.RAW);
    }


   /*.
    *  Shows the sequencer subroutines.
    */
    @Command(name="showsubs", description="Show the sequencer subroutines")
    public void showSubroutines() throws Exception
    {
        List<Map<String, Integer>> maps;
        maps = (List<Map<String, Integer>>)sendCommand(null, "getSubroutineMap");
        displayMaps("Subroutines", maps, Format.RAW);
    }


   /*.
    *  Shows the sequencer functionss.
    */
    @Command(name="showfunctions", description="Show the sequencer functions")
    public void showFunctions() throws Exception
    {
        List<Map<String, Integer>> maps;
        maps = (List<Map<String, Integer>>)sendCommand(null, "getFunctionMap");
        displayMaps("Functions", maps, Format.RAW);
    }


   /*.
    *  Shows the sequencer pointers.
    */
    @Command(name="showpointers", description="Show the sequencer pointers")
    public void showPointers() throws Exception
    {
        showPointers(Format.POINTER);
    }


   /*.
    *  Shows the sequencer pointers.
    */
    @Command(name="showpointers", description="Show the sequencer pointers")
    public void showPointers(@Argument(name="format", description="Display format")
                             Format format) throws Exception
    {
        List<Map<String, Integer>> maps;
        maps = (List<Map<String, Integer>>)sendCommand(null, "getPointers");
        displayMaps("Pointers", maps, format);
    }


   /*.
    *  Shows a sequencer parameter value.
    */
    @Command(name="showparameter", description="Show a parameter value")
    public void showParameter(@Argument(name="name", description="Parameter name")
                              String name) throws Exception
    {
        List<Integer> values = (List)sendCommand(null, "getParameter", name);
        out.print("Parameter " + name + ":");
        values.stream().forEach((value) -> {
            out.format(" %s", value);
        });
        out.println();
    }


   /*.
    *  Sets a sequencer parameter value.
    */
    @Command(name="setparameter", description="Set a parameter value")
    public void setParameter(@Argument(name="name", description="Parameter name")
                             String name,
                             @Argument(name="value", description="The value to set")
                             int value) throws Exception
    {
        sendCommand(null, "setParameter", name, value);
    }


   /*.
    *  Sets the sequencer data source.
    */
    @Command(name="setsource", description="Set the sequencer data source")
    public void setSource(@Argument(name="value", description="The value to set")
                          int value) throws Exception
    {
        sendCommand(null, "setDataSource", value);
    }


   /*.
    *  Starts the sequencer.
    */
    @Command(name="startsequencer", description="Start the sequencer")
    public void startSequencer() throws Exception
    {
        sendCommand(null, "startSequencer");
    }


   /*.
    *  Stops a sequencer loop.
    */
    @Command(name="stopsequencer", description="Stop a sequencer loop")
    public void stopSequencer() throws Exception
    {
        sendCommand(null, "sendStop");
    }


   /*.
    *  Steps a sequencer loop.
    */
    @Command(name="stepsequencer", description="Step a sequencer loop")
    public void stepSequencer() throws Exception
    {
        sendCommand(null, "sendStep");
    }


   /*.
    *  Saves the raw image to a directory.
    */
    @Command(name="saveimage", description="Save the raw image to a directory")
    public void saveImage(@Argument(name="dir", description="The name of the directory")
                          String dir) throws Exception
    {
        sendCommand(null, "saveImage", dir);
    }


   /*.
    *  Saves the raw image to the current directory.
    */
    @Command(name="saveimage", description="Save the raw image to the current directory")
    public void saveImage() throws Exception
    {
        saveImage("");
    }


   /*.
    *  Saves the image in FITS format.
    */
    @Command(name="savefitsimage", description="Save the image in FITS format")
    public void saveFitsImage(@Argument(name="file", description="The name of the file")
                              String file) throws Exception
    {
        sendCommand(null, "saveFitsImage", file);
    }


   /*.
    *  Dumps the contents of the current image.
    */
    @Command(name="dumpimage", description="Dump the contents of the current image")
    public void dumpImage(@Argument(name="offset", description="The offset to the first pixel")
                          int offset,
                          @Argument(name="size", description="The number of pixels")
                          int size) throws Exception
    {
        out.println(sendCommand(getReb(rebIndex), "getImage", offset, size));
    }


   /*.
    *  Dumps the contents of the current image.
    */
    @Command(name="dumpimage", description="Dump the contents of the current image")
    public void dumpImage(@Argument(description="The CCD number")
                          int ccd,
                          @Argument(description="The offset to the first pixel")
                          int offset,
                          @Argument(description="The number of pixels")
                          int size) throws Exception
    {
        out.println(sendCommand(getReb(rebIndex), "getImage", ccd, offset, size));
    }


   /*.
    *  Dumps the contents of a raw image file.
    *
    @Command(name="dumpimage", description="Dump the contents of a raw image file")
    public void dumpImage(@Argument(name="offset", description="The offset to the first pixel")
                          int offset,
                          @Argument(name="size", description="The number of pixels")
                          int size,
                          @Argument(name="file", description="The name of the file")
                          String file) throws Exception
    {
        out.println(sendCommand(null, "getImage", file, offset, size));
    }


   /*.
    *  Shows the metadata for the current image.
    */
    @Command(name="showimagestats", description="Show the metadata for the current image")
    public void showImageStats() throws Exception
    {
        out.println(sendCommand(getReb(rebIndex), "getImageMetadata"));
    }


   /*.
    *  Shows the REB statistics data.
    */
    @Command(name="showrebstats", description="Show the REB statistics data")
    public void showRebStats() throws Exception
    {
        out.println(sendCommand(getReb(rebIndex), "getRebStatus"));
    }


   /*.
    *  Saves the current configuration.
    */
    @Command(name="saveconfig", description="Save the current configuration")
    public void saveConfiguration() throws Exception
    {
        sendCommand(null, "saveConfig");
    }


   /*.
    *  Sends a synchronous command.
    */
    private Object sendCommand(String child, String func, Object... args)
        throws Exception
    {
        String dest = raftsSS + (child == null || child.isEmpty() ? "" : "/" + child);
        CommandRequest cmd = new CommandRequest(dest, func, args);
        return ca.sendSynchronousCommand(cmd, Duration.ofMillis(TIMEOUT));
    }


   /*.
    *  Receives status messages.
    *
    *  @param  s  The received message
    */
    @Override
    public void onStatusMessage(StatusMessage s)
    {
        if (!s.getOriginAgentInfo().getName().equals(raftsSS)
              || !(s instanceof StatusSubsystemData)) return;
        StatusSubsystemData sd = (StatusSubsystemData)s;
        if (!sd.getDataKey().equals(ImageState.KEY)) return;
        ImageState is = (ImageState)((KeyValueData)sd.getSubsystemData()).getValue();
        out.format("Received image: timestamp = %016x, length = %s\n",
                    is.getTimestamp(), is.getLength());
    }


   /*.
    *  Gets a REB device name given its index.
    */
    private String getReb(int index) throws Exception
    {
        List<String> devices = getRebs();
        for (String devc : devices) {
            if (index-- == 0) {
                return devc;
            }
        }
        throw new Exception("REB index out of range");
    }


   /*.
    *  Gets all REB device names.
    */
    private List<String> getRebs() throws Exception
    {
        return (List<String>)sendCommand(null, "getREBDeviceNames");
    }


   /*.
    *  Displays a time.
    */
    private static void displayTime(String desc, long time)
    {
        Date tm = new Date(time);
        out.format("%s time: %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL\n", desc, tm);
    }


   /*.
    *  Converts a returned value to a string.
    */
    private static String getString(Object value)
    {
        String sValue;
        String vType = value.getClass().getName();
        if (!vType.startsWith("[")) {
            sValue = value.toString();
        }
        else if (vType.endsWith("I")) {
            int[] iValue = (int[])value;
            StringBuilder text = new StringBuilder("[");
            text.append(iValue[0]);
            for (int k = 1; k < iValue.length; k++) {
                text.append(", ").append(iValue[k]);
            }
            text.append("]");
            sValue = text.toString();
        }
        else {
            double[] dValue = (double[])value;
            StringBuilder text = new StringBuilder("[");
            text.append(dValue[0]);
            for (int k = 1; k < dValue.length; k++) {
                text.append(", ").append(dValue[k]);
            }
            text.append("]");
            sValue = text.toString();
        }

        return sValue;
    }


   /*.
    *  Displays a set of named values.
    */
    private static void displayValues(Map<String, Object> values)
    {
        if (values.isEmpty()) return;
        out.print("Configured values:");
        int COL_SIZE = 39, posn = COL_SIZE + 1;
        String fmt = "\n%-" + COL_SIZE + "s";
        for (String name : values.keySet()) {
            String item = String.format("  %-20s: %s", name,
                                        getString(values.get(name)));
            if (posn > COL_SIZE || posn + item.length() > 2 * COL_SIZE) {
                out.format(fmt, item);
                posn = item.length() > COL_SIZE ? item.length() : COL_SIZE;
            }
            else {
                out.print(item);
                posn += item.length();
            }
        }
        out.println();
    }


   /*.
    *  Displays a list of maps.
    */
    private void displayMaps(String title, List<Map<String, Integer>> maps,
                             Format format) throws Exception
    {
        for (int j = 0; j < maps.size(); j++) {
            String head = getReb(j) + " " + title + ":";
            out.print(head);
            int leng = head.length();
            Map<String, Integer> map = maps.get(j);
            if (map == null || map.isEmpty()) {
                out.print(" None");
            }
            else {
                String pad = null;
                for (Map.Entry e : map.entrySet()) {
                    String item;
                    if (format == Format.POINTER) {
                        int value = (Integer)e.getValue();
                        item = " " + e.getKey() + "="
                                 + ptrTypeMap.get(value >> 8) + ":"
                                 + (value & 0xff);
                    }
                    else {
                        item = " " + e.getKey() + "=" + e.getValue();
                    }
                    if (leng + item.length() > 80) {
                        out.println();
                        if (pad == null) {
                            char[] padC = new char[head.length()];
                            Arrays.fill(padC, ' ');
                            pad = new String(padC);
                        }
                        out.print(pad);
                        leng = pad.length();
                    }
                    out.print(item);
                    leng += item.length();
                }
            }
            out.println();
        }
    }


   /*.
    *  Gets the model compiled from an XML sequencer file.
    */
    private FPGA2Model getModel(String file) throws Throwable
    {
        try {
            return (new FPGA2ModelBuilder()).compileFile(new File(file));
        }
        catch (Exception e) {
            Throwable t = e.getCause();
            throw t == null ? e : t;
        }
    }

}
