package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTabbedPane;
import javax.swing.text.NumberFormatter;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.DACS;
import org.lsst.ccs.subsystem.rafts.config.REB;

/**
 *  REB configuration editor.
 *
 *  @author  Eric Aubourg & Owen Saxton
 */
public class REBConfigPanel extends JPanel {

    private static final Map<Integer, String> dacNames = new HashMap<>();
    static {
        dacNames.put(DACS.SCLK_LOW, "SCLK Low");
        dacNames.put(DACS.SCLK_HIGH, "SCLK High");
        dacNames.put(DACS.PCLK_LOW, "PCLK Low");
        dacNames.put(DACS.PCLK_HIGH, "PCLK High");
        dacNames.put(DACS.RG_LOW, "RG Low");
        dacNames.put(DACS.RG_HIGH, "RG High");
        dacNames.put(DACS.SCLK_LOW_SH, "SCLK Low Sh");
        dacNames.put(DACS.SCLK_HIGH_SH, "SCLK High Sh");
        dacNames.put(DACS.PCLK_LOW_SH, "PCLK Low Sh");
        dacNames.put(DACS.PCLK_HIGH_SH, "PCLK High Sh");
        dacNames.put(DACS.RG_LOW_SH, "RG Low Sh");
        dacNames.put(DACS.RG_HIGH_SH, "RG High Sh");
        dacNames.put(DACS.CSGATE_1, "CS Gate 0");
        dacNames.put(DACS.CSGATE_2, "CS Gate 1");
        dacNames.put(DACS.CSGATE_3, "CS Gate 2");
    }
    
    private static final List<Integer> dacList1 = new ArrayList<>();
    static {
        dacList1.add(DACS.SCLK_LOW);
        dacList1.add(DACS.SCLK_HIGH);
        dacList1.add(DACS.PCLK_LOW);
        dacList1.add(DACS.PCLK_HIGH);
        dacList1.add(DACS.RG_LOW);
        dacList1.add(DACS.RG_HIGH);
    }

    private static final List<Integer> dacList2 = new ArrayList<>();
    static {
        dacList2.add(DACS.SCLK_LOW);
        dacList2.add(DACS.SCLK_LOW_SH);
        dacList2.add(DACS.SCLK_HIGH);
        dacList2.add(DACS.SCLK_HIGH_SH);
        dacList2.add(DACS.PCLK_LOW);
        dacList2.add(DACS.PCLK_LOW_SH);
        dacList2.add(DACS.PCLK_HIGH);
        dacList2.add(DACS.PCLK_HIGH_SH);
        dacList2.add(DACS.RG_LOW);
        dacList2.add(DACS.RG_LOW_SH);
        dacList2.add(DACS.RG_HIGH);
        dacList2.add(DACS.RG_HIGH_SH);
    }

    private static final List<Integer> dacList3 = new ArrayList<>();
    static {
        dacList3.add(DACS.SCLK_LOW);
        dacList3.add(DACS.SCLK_LOW_SH);
        dacList3.add(DACS.SCLK_HIGH);
        dacList3.add(DACS.PCLK_LOW);
        dacList3.add(DACS.PCLK_LOW_SH);
        dacList3.add(DACS.PCLK_HIGH);
        dacList3.add(DACS.RG_LOW);
        dacList3.add(DACS.RG_LOW_SH);
        dacList3.add(DACS.RG_HIGH);
    }

    private static final int TF_WIDTH = 50, TF_WIDTH_L = 80, TF_WIDTH_S = 30;
    private static final Map<String, JDialog> active = new HashMap<>();
    private static final NumberFormatter fmt5 = new NumberFormatter(new DecimalFormat("####0"));
    private static final NumberFormatter fmtf5 = new NumberFormatter(new DecimalFormat("##0.0#"));
    private static final NumberFormatter fmt3 = new NumberFormatter(new DecimalFormat("##0"));
    private static final Insets insets_std = new Insets(4, 4, 4, 4);
    private static final Insets insets_big = new Insets(14, 4, 6, 4);

    private final JLabel lbIdDesc = new JLabel("REB id:");
    private final JLabel lblId = new JLabel("X");
    private final JLabel lbIfcNameDesc = new JLabel("REB iface:");
    private final JLabel lblIfcName = new JLabel("X");
    private final JLabel lbCcdMaskDesc = new JLabel("CCD mask:");
    private final JLabel lblCcdMask = new JLabel("X");
    private final JLabel[] lbDacVoltage = new JLabel[DACS.NUM_DACS];
    private final JFormattedTextField[] tfDacVoltage = new JFormattedTextField[DACS.NUM_DACS];
    private final JLabel[] lbDacCurrent = new JLabel[3];
    private final JFormattedTextField[] tfDacCurrent = new JFormattedTextField[3];
    private final ASPICConfigPanel[] aspicPanels = new ASPICConfigPanel[REB.N_ASIC];
    private final BiasesConfigPanel[] biasPanels = new BiasesConfigPanel[REB.N_CCD];
    private final JPanel[] ccdPanels = new JPanel[REB.N_CCD];
    private final JTabbedPane ccdTabs = new JTabbedPane();
    private final JPanel ccdContainer = new JPanel();

    private final REB reb;
    private final List<Integer> dacs, ccds = new ArrayList<>();
    private final boolean dacRaw, showCsgate;

    /**
     *  Parent interface.
     */
    public interface Parent {

        Object sendCommand(String rebName, String command, Object... params);

        void display(String message);
    }

    /**
     *  Constructor.
     *
     *  @param  reb     The REB configuration data
     *  @param  name    The REB name
     *  @param  parent  The parent, which provides command & message services
     */
    public REBConfigPanel(REB reb, String name, Parent parent) {

        this.reb = reb;

        dacRaw = reb.isDacRaw();
        int version = reb.getDacVersion();
        dacs = (!dacRaw || version == REB.VERSION_0) ? dacList1
                  : version == REB.VERSION_1 || version == REB.VERSION_3 ? dacList2 : dacList3;
        showCsgate = version == REB.VERSION_0;
        boolean groupByThree = dacRaw && version == REB.VERSION_2;

        fmt3.setValueClass(Integer.class);
        fmt5.setValueClass(Integer.class);
        fmtf5.setValueClass(Double.class);

        /*
         *  Get mask of active CCDs and generate list
         */
        int ccdMask = reb.getMaxCcdMask();
        for (int j = 0; j < 3; j++) {
            if (((1 << j) & ccdMask) != 0) {
                ccds.add(j);
            }
        }

        /*
         *  Create text fields, etc, and set their properties
         */
        Dimension d3 = lblId.getPreferredSize();
        d3.width = TF_WIDTH_S;
        lblId.setPreferredSize(d3);
        lblId.setMinimumSize(d3);

        d3.width = TF_WIDTH_L;
        lblIfcName.setPreferredSize(d3);
        lblIfcName.setMinimumSize(d3);

        d3.width = TF_WIDTH_S;
        lblCcdMask.setPreferredSize(d3);
        lblCcdMask.setMinimumSize(d3);

        Dimension d5 = (new JFormattedTextField()).getPreferredSize();
        d5.width = TF_WIDTH;
        for (int dac : dacs) {
            lbDacVoltage[dac] = new JLabel(dacNames.get(dac) + ":");
            tfDacVoltage[dac] = new JFormattedTextField(dacRaw ? fmt5 : fmtf5);
            tfDacVoltage[dac].setPreferredSize(d5);
            tfDacVoltage[dac].setMinimumSize(d5);
        }

        if (showCsgate) {
            for (int ccd = 0; ccd < 3; ccd++) {
                lbDacCurrent[ccd] = new JLabel("CS Gate " + ccd + ":");
                tfDacCurrent[ccd] = new JFormattedTextField(fmt5);
                tfDacCurrent[ccd].setPreferredSize(d5);
                tfDacCurrent[ccd].setMinimumSize(d5);
            }
        }

        /*
         *  Add graphical components to the panel
         */
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets_std;

        // REB ID
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(lbIdDesc, gbc);
        gbc.gridx++;
        add(lblId, gbc);

        // REB interface name
        gbc.gridx = 0;
        gbc.gridy++;
        add(lbIfcNameDesc, gbc);
        gbc.gridx++;
        gbc.gridwidth = 3;
        add(lblIfcName, gbc);
        gbc.gridwidth = 1;

        // CCD mask
        gbc.gridx = 0;
        gbc.gridy++;
        add(lbCcdMaskDesc, gbc);
        gbc.gridx++;
        add(lblCcdMask, gbc);

        // DAC voltages
        gbc.gridx = 0;
        gbc.gridy++;
        gbc.insets = insets_big;
        add(new JLabel("DAC voltages"), gbc);
        gbc.gridx++;
        gbc.gridwidth = 3;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.CENTER;
        add(new JSeparator(), gbc);
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets_std;

        int fld = 0;
        for (int dac : dacs) {
            gbc.gridx = 2 * (fld & 1);
            gbc.gridy += 1 - (fld & 1);
            add(lbDacVoltage[dac], gbc);
            gbc.gridx++;
            add(tfDacVoltage[dac], gbc);
            fld++;
            if (groupByThree && (fld & 3) == 3) {
                fld++;
            }
        }

        // DAC currents
        if (showCsgate) {
            gbc.gridx = 0;
            gbc.gridy++;
            gbc.insets = insets_big;
            add(new JLabel("DAC currents"), gbc);
            gbc.gridx++;
            gbc.gridwidth = 3;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.anchor = GridBagConstraints.CENTER;
            add(new JSeparator(), gbc);
            gbc.gridwidth = 1;
            gbc.fill = GridBagConstraints.NONE;
            gbc.anchor = GridBagConstraints.NORTHWEST;
            gbc.insets = insets_std;

            for (int ccd : ccds) {
                gbc.gridx = 0;
                gbc.gridy++;
                add(lbDacCurrent[ccd], gbc);
                gbc.gridx++;
                add(tfDacCurrent[ccd], gbc);
            }
        }

        // Vertical separator
        gbc.gridx = 4;
        gbc.gridy = 0;
        gbc.fill = GridBagConstraints.VERTICAL;
        gbc.gridheight = GridBagConstraints.REMAINDER;
        gbc.gridwidth = 1;
        add(new JSeparator(JSeparator.VERTICAL), gbc);
        gbc.fill = GridBagConstraints.NONE;

        // CCD container
        gbc.gridx = 5;
        gbc.gridy = 0;
        gbc.gridheight = GridBagConstraints.REMAINDER;
        gbc.gridwidth = GridBagConstraints.REMAINDER;
        add(ccdContainer, gbc);

        // CCD tabbed panel
        ccdContainer.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        gbc1.gridwidth = 2;
        gbc1.gridheight = 1;
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        gbc1.anchor = GridBagConstraints.NORTHWEST;
        ccdContainer.add(ccdTabs, gbc1);

        // Set up each CCD panel
        for (int ccd : ccds) {
            JPanel ccdPanel = new JPanel();
            ccdPanel.setLayout(new GridBagLayout());
            GridBagConstraints gbc2 = new GridBagConstraints();
            gbc2.insets = insets_std;
            gbc2.gridwidth = 1;
            gbc2.gridheight = 1;
            gbc2.anchor = GridBagConstraints.NORTHWEST;
            gbc2.gridx = 0;
            gbc2.gridy = 0;
            for (int side = 0; side < 2; side++) {
                int jc = 2 * ccd + side;
                ASPIC aspic = reb.getAspics()[jc];
                if (aspic != null) {
                    aspicPanels[jc] = new ASPICConfigPanel(jc, aspic);
                    ccdPanel.add(aspicPanels[jc], gbc2);
                }
                gbc2.gridy++;
            }
            BiasDACS bias = reb.getBiases()[ccd];
            if (bias != null) {
                biasPanels[ccd] = new BiasesConfigPanel(ccd, bias, reb.isBiasRaw());
                ccdPanel.add(biasPanels[ccd], gbc2);
            }
            ccdPanels[ccd] = ccdPanel;
            ccdTabs.addTab("CCD " + ccd, ccdPanels[ccd]);
        }

        read();
    }


   /**
    *  Saves new values to the REB data.
    */
    private void save() {

        if (dacRaw) {
            int[] values = reb.getDacs().getValues();
            for (int dac : dacs) {
                values[dac] = (Integer)tfDacVoltage[dac].getValue();
            }
            if (showCsgate) {
                for (int ccd : ccds) {
                    int dac = ccd == 0 ? DACS.CSGATE_1 : ccd == 1 ? DACS.CSGATE_2 : DACS.CSGATE_3;
                    values[dac] = (Integer)tfDacCurrent[ccd].getValue();
                }
            }
        }
        else {
            double[] values = reb.getDacs().getPValues();
            for (int dac : dacs) {
                values[dac] = (Double)tfDacVoltage[dac].getValue();
            }
        }
        for (ASPICConfigPanel panel : aspicPanels) {
            if (panel != null) {
                panel.save();
            }
        }
        for (BiasesConfigPanel panel : biasPanels) {
            if (panel != null) {
                panel.save();
            }
        }
    }


   /**
    *  Reads REB data and populates graphical elements.
    */
    private void read() {

        lblId.setText(Integer.toString(reb.getId()));
        lblIfcName.setText(reb.getIfcName());
        lblCcdMask.setText(Integer.toString(reb.getCcdMask()));
        if (dacRaw) {
            int values[] = reb.getDacs().getValues();
            for (int dac : dacs) {
                tfDacVoltage[dac].setValue(values[dac]);
            }
            if (showCsgate) {
                for (int ccd : ccds) {
                    int ix = ccd == 0 ? DACS.CSGATE_1 : ccd == 1 ? DACS.CSGATE_2 : DACS.CSGATE_3;
                    tfDacCurrent[ccd].setValue(values[ix]);
                }
            }
        }
        else {
            double values[] = reb.getDacs().getPValues();
            for (int dac : dacs) {
                tfDacVoltage[dac].setValue(values[dac]);
            }
        }
        for (ASPICConfigPanel panel : aspicPanels) {
            if (panel != null) {
                panel.read();
            }
        }
        for (BiasesConfigPanel panel : biasPanels) {
            if (panel != null) {
                panel.read();
            }
        }
    }


   /**
    *  Edits the REB configuration.
    *
    *  @param  pWindow  The parent window
    *  @param  name     The REB name
    *  @param  parent   The Parent interface implementer
    */
    public static void edit(Window pWindow, String name, Parent parent) {

        JDialog dlg0 = active.get(name);
        if (dlg0 != null) {
            dlg0.toFront();
            return;
        }
        final REB reb = (REB)parent.sendCommand(name, "getREBConfig");
        if (reb == null) return;
        final JDialog dlg = new JDialog(pWindow, name + " Configuration",
                                        Dialog.ModalityType.MODELESS);
        active.put(name, dlg);
        final REBConfigPanel cfgPanel = new REBConfigPanel(reb, name, parent);

        JButton btLoadDacs = new JButton("Load DACs");
        btLoadDacs.setFocusable(false);
        btLoadDacs.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Object reply = parent.sendCommand(name, "loadDacs", true);
                if (reply != null) {
                    Integer count = (Integer)reply;
                    parent.display(count + " " + name + " DACs loaded");
                }
                reply = parent.sendCommand(name, "loadBiasDacs", true);
                if (reply != null) {
                    Integer count = (Integer)reply;
                    parent.display(count + " " + name + " bias DACs loaded");
                }
            }
        });

        JButton btLoadAspics = new JButton("Load ASPICs");
        btLoadAspics.setFocusable(false);
        btLoadAspics.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Integer count = (Integer)parent.sendCommand(name, "loadAspics", true);
                if (count == null) return;
                Integer mask = (Integer)parent.sendCommand(name, "checkAspics");
                List<Integer> masks = null;
                if (mask != null) {
                    masks = new ArrayList<>();
                    masks.add(mask);
                }
                parent.display(count + " " + name + " ASPICs loaded: "
                                 + getResult(masks));
            }
        });

        JButton btUndo = new JButton("Undo");
        btUndo.setFocusable(false);
        btUndo.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                cfgPanel.read();
            }
        });

        JButton btApply = new JButton("Apply");
        btApply.setFocusable(false);
        btApply.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                cfgPanel.save();
                Object reply = parent.sendCommand(name, "setREBConfig", reb);
                if (reply != null) {
                    parent.display(name + " configuration applied");
                }
                REB newReb = (REB)parent.sendCommand(name, "getREBConfig");
                if (newReb == null) {
                    active.remove(name);
                    dlg.setVisible(false);
                    dlg.dispose();
                }
                reb.copyFrom(newReb);
                cfgPanel.read();
            }
        });

        JButton btClose = new JButton("Close");
        btClose.setFocusable(false);
        btClose.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                active.remove(name);
                dlg.setVisible(false);
                dlg.dispose();
            }
        });

        JButton btOK = new JButton("OK");
        btOK.setFocusable(false);
        btOK.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                cfgPanel.save();
                Object reply = parent.sendCommand(name, "setREBConfig", reb);
                active.remove(name);
                dlg.setVisible(false);
                dlg.dispose();
            }
        });

        JPanel btnPanel = new JPanel();
        btnPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 4);
        btnPanel.add(btLoadDacs, gbc);
        btnPanel.add(btLoadAspics, gbc);
        gbc.insets.left = 20;
        btnPanel.add(btUndo, gbc);
        gbc.insets.left = 4;
        btnPanel.add(btApply, gbc);
        gbc.insets.left = 20;
        btnPanel.add(btOK, gbc);
        gbc.insets.left = 4;
        btnPanel.add(btClose, gbc);

        dlg.add(cfgPanel, BorderLayout.CENTER);
        dlg.add(btnPanel, BorderLayout.SOUTH);
        dlg.pack();
        dlg.setLocationRelativeTo(pWindow);
        dlg.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                active.remove(name);
                dlg.setVisible(false);
                dlg.dispose();
            }
        });
        dlg.setVisible(true);
    }


   /**
    *  Forms a result string from a list of mismatch masks.
    */
    private static String getResult(List<Integer> masks) {
        if (masks == null) return "check not done";
        int sum = 0;
        for (int mask : masks) {
            sum |= mask;
        }
        if (sum == 0) return "no errors";
        StringBuilder result = new StringBuilder("mismatches =");
        for (int mask : masks) {
            result.append(String.format(" %08x", mask));
        }
        return result.toString();
    }


   /**
    *  Test program.
    */
    public static void main(String[] args) throws Exception {
        final REB r;

        final File file = new File("reb.ser");
        if (file.exists()) {
            ObjectInputStream is = new ObjectInputStream(
                    new BufferedInputStream(new FileInputStream(file)));
            r = (REB) is.readObject();
            is.close();
        } else {
            r = new REB();
        }

        // UIManager.setLookAndFeel(UIManager
        // .getCrossPlatformLookAndFeelClassName());

        final REBConfigPanel p = new REBConfigPanel(r, null, null);

        final JFrame f = new JFrame("REB");

        JPanel pane = new JPanel();
        pane.setLayout(new BorderLayout());
        pane.add(p, BorderLayout.CENTER);

        JPanel btPanel = new JPanel();
        JButton btClose = new JButton("Close");
        btClose.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                f.dispose();
            }
        });
        JButton btOK = new JButton("OK");
        btOK.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                p.save();
                try {
                    ObjectOutputStream s = new ObjectOutputStream(
                            new BufferedOutputStream(new FileOutputStream(file)));
                    s.writeObject(r);
                    s.close();
                } catch (FileNotFoundException e1) {
                    // TODO Auto-generated catch block
                    e1.printStackTrace();
                } catch (IOException e1) {
                    // TODO Auto-generated catch block
                    e1.printStackTrace();
                }
                f.dispose();
            }
        });

        btPanel.add(btClose);
        btPanel.add(btOK);

        pane.add(btPanel, BorderLayout.SOUTH);

        JScrollPane scroll = new JScrollPane(pane);
        f.setContentPane(scroll);

        f.pack();
        f.setVisible(true);
    }

    private static final long serialVersionUID = 1L;
}
