package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.text.NumberFormatter;
import org.lsst.ccs.subsystem.rafts.config.CABAC;
import org.lsst.ccs.subsystem.rafts.config.REB;

/**
 ****************************************
 *
 *  CABAC configuration editor.
 *
 *  @author  Eric Aubourg & Owen Saxton
 *
 ****************************************
 */
class CABACConfigPanel extends JPanel {

    static class LabelDesc {

        String text;
        int x;
        int y;

        LabelDesc(String text, int x, int y) {
            this.text = text;
            this.x = x;
            this.y = y;
        }

    }

    static class FieldDesc {

        int index;
        int x;
        int y;

        FieldDesc(int index, int x, int y) {
            this.index = index;
            this.x = x;
            this.y = y;
        }

    }

    private static final Insets INSETS_STD = new Insets(4, 4, 4, 4);
    private static final int TF_WIDTH_S = 36;

    final int iCabac;
    final CABAC cabac;
    private final JFormattedTextField[] fields;

    private static final List<LabelDesc> labelDescs1 = new ArrayList<>();
    private static final List<FieldDesc> fieldDescs1 = new ArrayList<>();
    static {
        labelDescs1.add(new LabelDesc("RD:", 0, 0));
        labelDescs1.add(new LabelDesc("OG:", 0, 1));
        labelDescs1.add(new LabelDesc("GD:", 0, 2));
        labelDescs1.add(new LabelDesc("RG:", 0, 3));
        labelDescs1.add(new LabelDesc("SCLKs:", 0, 4));
        labelDescs1.add(new LabelDesc("PCLKs:", 0, 5));
        labelDescs1.add(new LabelDesc("Rdout OD:", 0, 6));
        labelDescs1.add(new LabelDesc("Expos OD:", 0, 7));
        fieldDescs1.add(new FieldDesc(CABAC.RD, 1, 0));
        fieldDescs1.add(new FieldDesc(CABAC.OG, 1, 1));
        fieldDescs1.add(new FieldDesc(CABAC.GD, 1, 2));
        fieldDescs1.add(new FieldDesc(CABAC.RG_RISE, 1, 3));
        fieldDescs1.add(new FieldDesc(CABAC.SCLK0_RISE, 1, 4));
        fieldDescs1.add(new FieldDesc(CABAC.SCLK1_RISE, 2, 4));
        fieldDescs1.add(new FieldDesc(CABAC.SCLK2_RISE, 3, 4));
        fieldDescs1.add(new FieldDesc(CABAC.PCLK0_RISE, 1, 5));
        fieldDescs1.add(new FieldDesc(CABAC.PCLK1_RISE, 2, 5));
        fieldDescs1.add(new FieldDesc(CABAC.PCLK2_RISE, 3, 5));
        fieldDescs1.add(new FieldDesc(CABAC.PCLK3_RISE, 4, 5));
        fieldDescs1.add(new FieldDesc(CABAC.OD0_RDO, 1, 6));
        fieldDescs1.add(new FieldDesc(CABAC.OD1_RDO, 2, 6));
        fieldDescs1.add(new FieldDesc(CABAC.OD0_EXP, 1, 7));
        fieldDescs1.add(new FieldDesc(CABAC.OD1_EXP, 2, 7));
    }

    private static final List<LabelDesc> labelDescs2 = new ArrayList<>();
    private static final List<FieldDesc> fieldDescs2 = new ArrayList<>();
    static {
        labelDescs2.add(new LabelDesc("RD:", 0, 0));
        labelDescs2.add(new LabelDesc("OG:", 0, 1));
        labelDescs2.add(new LabelDesc("GD:", 0, 2));
        labelDescs2.add(new LabelDesc("RG rise:", 0, 3));
        labelDescs2.add(new LabelDesc("RG fall:", 0, 4));
        labelDescs2.add(new LabelDesc("SCLKs rise:", 0, 5));
        labelDescs2.add(new LabelDesc("SCLKs fall:", 0, 6));
        labelDescs2.add(new LabelDesc("PCLKs rise:", 0, 7));
        labelDescs2.add(new LabelDesc("PCLKs fall:", 0, 8));
        labelDescs2.add(new LabelDesc("Rdout OD:", 0, 9));
        labelDescs2.add(new LabelDesc("Expos OD:", 0, 10));
        fieldDescs2.add(new FieldDesc(CABAC.RD, 1, 0));
        fieldDescs2.add(new FieldDesc(CABAC.OG, 1, 1));
        fieldDescs2.add(new FieldDesc(CABAC.GD, 1, 2));
        fieldDescs2.add(new FieldDesc(CABAC.RG_RISE, 1, 3));
        fieldDescs2.add(new FieldDesc(CABAC.RG_FALL, 1, 4));
        fieldDescs2.add(new FieldDesc(CABAC.SCLK0_RISE, 1, 5));
        fieldDescs2.add(new FieldDesc(CABAC.SCLK1_RISE, 2, 5));
        fieldDescs2.add(new FieldDesc(CABAC.SCLK2_RISE, 3, 5));
        fieldDescs2.add(new FieldDesc(CABAC.SCLK0_FALL, 1, 6));
        fieldDescs2.add(new FieldDesc(CABAC.SCLK1_FALL, 2, 6));
        fieldDescs2.add(new FieldDesc(CABAC.SCLK2_FALL, 3, 6));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK0_RISE, 1, 7));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK1_RISE, 2, 7));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK2_RISE, 3, 7));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK3_RISE, 4, 7));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK0_FALL, 1, 8));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK1_FALL, 2, 8));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK2_FALL, 3, 8));
        fieldDescs2.add(new FieldDesc(CABAC.PCLK3_FALL, 4, 8));
        fieldDescs2.add(new FieldDesc(CABAC.OD0_RDO, 1, 9));
        fieldDescs2.add(new FieldDesc(CABAC.OD1_RDO, 2, 9));
        fieldDescs2.add(new FieldDesc(CABAC.OD0_EXP, 1, 10));
        fieldDescs2.add(new FieldDesc(CABAC.OD1_EXP, 2, 10));
    }


   /**
    *  Constructor.
    */
    CABACConfigPanel(int vrsn, int ic, CABAC c) {

        iCabac = ic;
        cabac = c;

        NumberFormatter fmt3 = new NumberFormatter(new DecimalFormat("##0"));
        fmt3.setValueClass(Integer.class);

        fields = new JFormattedTextField[CABAC.NUM_VALUES];
        Dimension d = null;
        for (int j = 0; j < fields.length; j++) {
            JFormattedTextField field = new JFormattedTextField(fmt3);
            if (j == 0) {
                d = field.getPreferredSize();
                d.width = TF_WIDTH_S;
            }
            field.setPreferredSize(d);
            field.setMinimumSize(d);
            fields[j] = field;
        }

        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = INSETS_STD;
        gbc.anchor = GridBagConstraints.NORTHWEST;

        String title = "CABAC " + iCabac
                         + (iCabac % 2 == 0 ? " (upper)" : " (lower)");
        setBorder(BorderFactory.createTitledBorder(title));

        for (LabelDesc ld : vrsn == REB.VERSION_0 ? labelDescs1 : labelDescs2) {
            gbc.gridx = ld.x;
            gbc.gridy = ld.y;
            add(new JLabel(ld.text), gbc);
        }

        for (FieldDesc fd : vrsn == REB.VERSION_0 ? fieldDescs1 : fieldDescs2) {
            gbc.gridx = fd.x;
            gbc.gridy = fd.y;
            add(fields[fd.index], gbc);
        }
    }


   /**
    *  Saves new values to the CABAC data.
    */
    void save() {

        int[] values = cabac.getValues();
        for (int j = 0; j < CABAC.NUM_VALUES; j++) {
            values[j] = (Integer)fields[j].getValue();
        }
    }


   /**
    *  Reads values from the CABAC data and populates fields.
    */
    void read() {

        int[] values = cabac.getValues();
        for (int j = 0; j < CABAC.NUM_VALUES; j++) {
            fields[j].setValue(values[j]);
        }
    }

    private static final long serialVersionUID = 1L;
}
