package org.lsst.ccs.subsystem.rafts.fpga.compiler;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.lsst.ccs.subsystem.rafts.fpga.xml.Sequencer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.SequencerTextParser;
import org.xml.sax.SAXException;

/**
 * The FPGA-specific visitor.
 *
 * @author aubourg
 *
 * Creates a low level model, and converts it to a list of FPGA opcodes.
 *
 */
public class FPGA2ModelBuilder {

    /**
     * Read the specified input stream as either an .xml or .seq file. This
     * method examines the contents of the input stream to determine if it is an
     * xml or seq file.
     *
     * @param inputStream The input stream to read
     * @return The FPGA2Model constructed from the input steam
     * @throws IOException If an IO exception occurs while reading the file
     * @throws RuntimeException If a file format error is detected.
     */
    public FPGA2Model compileFile(InputStream inputStream) throws IOException {
        return compileFile(inputStream, null);
    }

    /**
     * Read the specified input stream as either an .xml or .seq file. This
     * method examines the contents of the input stream to determine if it is an
     * xml or seq file.
     *
     * @param inputStream The input stream to read
     * @param clockPeriodOverride If not null, overrides and clock period specified in the sequencer file
     * @return The FPGA2Model constructed from the input steam
     * @throws IOException If an IO exception occurs while reading the file
     * @throws RuntimeException If a file format error is detected.
     */
    public FPGA2Model compileFile(InputStream inputStream, BigDecimal clockPeriodOverride) throws IOException {
        if (inputStream == null) {
            throw new IOException("Input stream is null");
        }
        BufferedInputStream in = inputStream instanceof BufferedInputStream ? (BufferedInputStream) inputStream : new BufferedInputStream(inputStream);
        // mark is guaranteed to be supported by BufferedInputStream
        in.mark(1);
        //XML files must always begin with <?xml while SEQ files never do
        boolean isXml = in.read() == '<';
        in.reset();
        Sequencer s;
        if (isXml) {
            try {
                JAXBContext context = JAXBContext.newInstance(Sequencer.class);
                Unmarshaller u = context.createUnmarshaller();

                try (java.io.InputStream si = Sequencer.class.getResourceAsStream("schema.xsd")) {
                    SchemaFactory sf = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
                    Schema schema = sf.newSchema(new StreamSource(si));
                    u.setSchema(schema);
                }

                s = (Sequencer) u.unmarshal(in);
            } catch (JAXBException e) {
                throw new RuntimeException("error reading XML file", e);
            } catch (SAXException e) {
                throw new RuntimeException("error validating XML file", e);
            }
        } else {
            s = SequencerTextParser.parse(new InputStreamReader(in));
        }
        FPGA2ModelBuilderVisitor visitor = new FPGA2ModelBuilderVisitor(clockPeriodOverride);
        visitor.visit(s);
        s.validate();
        visitor.getModel().validate();
        return visitor.getModel();
    }

    /**
     * Read a file and parse it as either a .xml or .seq file.
     *
     * @param srcFile The file to read
     * @return The model extracted from the file
     * @throws IOException If an IO exception occurs while reading the file
     * @throws RuntimeException If a file format error is detected.
     */
    public FPGA2Model compileFile(File srcFile) throws IOException {
        try (InputStream in = new FileInputStream(srcFile)) {
            return compileFile(in);
        }
    }

}
