package org.lsst.ccs.subsystem.rafts.fpga.compiler;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.lsst.ccs.drivers.reb.Sequencer;

import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo.PointerKind;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Call;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Callable;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Channel;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Function;
import org.lsst.ccs.subsystem.rafts.fpga.xml.FunctionPointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Main;
import org.lsst.ccs.subsystem.rafts.fpga.xml.RepeatFunctionPointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.RepeatSubroutinePointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Subroutine;
import org.lsst.ccs.subsystem.rafts.fpga.xml.SubroutinePointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Timeslice;

// Old version:
// opcode on 4 bits
// 0x1: function exe  + 4 bits function ID + 1 bit infinite loop + 23 bits repetition
// 0x2: jump to address  + 10 bits address + 1 bit infinite loop + 17 bits repetition
// 0x3: subroutine trailer
// 0x4: end of program  
// New version (REB 3):
// 0x1 – function execution;
// 0x2 – function execution with function pointer;
// 0x3 – function execution with repetitions pointer;
// 0x4 – function execution with function and repetitions pointers;
// 0x5 – jump to address;
// 0x6 – jump to address with address pointer;
// 0x7 – jump to address with repetitions pointer;
// 0x8 – jump to address with address and repetitions pointer;
// 0xE – subroutine trailer;
// 0xF – end of program;
// TODO separate the model and its builder?
/**
 * The FPGA-specific model.
 *
 * @author aubourg
 *
 *
 */
public class FPGA2Model implements Serializable {

    public static final int CMD_LINES = 0, CMD_TIME = 1, CMD_PROGFUNC = 2, CMD_PROGJUMP = 3, CMD_PROGSUBE = 4,
            CMD_PROGEND = 5, CMD_PROGFUNC_FP = 6, // (function pointer)
            CMD_PROGFUNC_RP = 7, // (repeat count pointer)
            CMD_PROGFUNC_FRP = 8, // (function and repeat count pointers)
            CMD_PROGJUMP_AP = 9, // (address pointer)
            CMD_PROGJUMP_RP = 10, // (repeat count pointer)
            CMD_PROGJUMP_ARP = 11, // (address and repeat count pointers)
            CMD_POINTER_FA = 12, // function address pointer
            CMD_POINTER_FR = 13, // function repeat count pointer
            CMD_POINTER_SA = 14, // subroutine address pointer
            CMD_POINTER_SR = 15; // subroutine repeat count pointer

    private static final long serialVersionUID = 3694475501274351773L;

    final Map<Integer, RegisterOpcode> timing = new TreeMap<>();
    // List of all routines. By routines we mean subroutines and functions
    final List<FPGARoutine> routines = new ArrayList<>();
    final Map<Subroutine, FPGARoutine> routinesMap = new LinkedHashMap<>();
    final Map<Function, Integer> functionsMap = new LinkedHashMap<>();
    final Map<SubroutinePointer, Integer> routinesPtrMap = new LinkedHashMap<>();
    final Map<FunctionPointer, Integer> functionsPtrMap = new LinkedHashMap<>();
    final Map<RepeatSubroutinePointer, Integer> routinesRepPtrMap = new LinkedHashMap<>();
    final Map<RepeatFunctionPointer, Integer> functionsRepPtrMap = new LinkedHashMap<>();
    int lastAddr;
    private final Map<String, String> metadata = new LinkedHashMap<>();
    private final Map<String, Integer> channels = new LinkedHashMap<>();

    public static abstract class AddressAndValue implements Serializable {

        private static final long serialVersionUID = -1583904741063682560L;

        public abstract int getAddress();

        public abstract int getValue();
    }

    private static abstract class RegisterOpcode extends AddressAndValue {

        public abstract int[] getCommand();

        public abstract String getComment();

        public void dump() {
            System.out.printf("%08x %08x // %s\n", getAddress(), getValue(), getComment());
        }
    }

    static class SliceValues extends RegisterOpcode implements Serializable {

        private static final long serialVersionUID = -4763340493598561416L;

        private final int functionIndex;
        private final Function function;
        private final Timeslice s;
        private List<Integer> upConstant = new ArrayList<>();

        SliceValues(int functionIndex, Function function, Timeslice s) {
            super();
            this.functionIndex = functionIndex;
            this.function = function;
            this.s = s;
            function.getConstants().stream().filter((c) -> (c.getValue().equals("1"))).forEach((c) -> {
                upConstant.add(c.getChannel().getValue());
            });
        }

        @Override
        public int getAddress() {
            return 0x100000 | (functionIndex << 4) | s.getIndex();
        }

        @Override
        public int getValue() {
            int w = 0;
            for (int jc : upConstant) {
                w |= 1 << jc;
            }
            for (Channel c : s.getUpChannels()) {
                w |= (1 << c.getValue());
            }
            return w;
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[upConstant.size() + s.getUpChannels().size() + 3];
            int j = 0;
            cmnd[j++] = CMD_LINES;
            cmnd[j++] = functionIndex;
            cmnd[j++] = s.getIndex();
            for (int jc : upConstant) {
                cmnd[j++] = jc;
            }
            for (Channel c : s.getUpChannels()) {
                cmnd[j++] = c.getValue();
            }
            return cmnd;
        }

        @Override
        public String getComment() {
            return String.format("%-20s  @%-5d %-5d : %s", function.getId(), s.getStartNanos(), s.getDurationNanos(),
                    s.getValue());
        }
    }

    static class SliceTime extends RegisterOpcode implements Serializable {

        private static final long serialVersionUID = -386728319462273830L;

        private final int functionIndex;
        private final Function function;
        private final Timeslice s;

        SliceTime(int functionIndex, Function function, Timeslice s) {
            super();
            this.functionIndex = functionIndex;
            this.function = function;
            this.s = s;
        }

        @Override
        public int getAddress() {
            return 0x200000 | (functionIndex << 4) | s.getIndex();
        }

        @Override
        public int getValue() {
            return s.getDurationNanos() / 10;
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_TIME;
            cmnd[1] = functionIndex;
            cmnd[2] = s.getIndex();
            cmnd[3] = s.getDurationNanos() / 10;
            return cmnd;
        }

        @Override
        public String getComment() {
            return String.format("%-20s  @%-5d %-5d : %s", function.getId(), s.getStartNanos(), s.getDurationNanos(),
                    s.getValue());
        }
    }

    static class EndSliceTime extends RegisterOpcode implements Serializable {

        private static final long serialVersionUID = 2688075986647178842L;

        private final int functionIndex;
        private final Function function;
        private final int sliceIndex;

        EndSliceTime(int functionIndex, Function function, int sliceIndex) {
            super();
            this.functionIndex = functionIndex;
            this.function = function;
            this.sliceIndex = sliceIndex;
        }

        @Override
        public int getAddress() {
            return 0x200000 | (functionIndex << 4) | sliceIndex;
        }

        @Override
        public int getValue() {
            return 0;
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_TIME;
            cmnd[1] = functionIndex;
            cmnd[2] = sliceIndex;
            cmnd[3] = 0;
            return cmnd;
        }

        @Override
        public String getComment() {

            return String.format("%-20s  end", function.getId());
        }
    }

    static abstract class StackOpCode extends RegisterOpcode implements Serializable {

        private static final long serialVersionUID = -7599004220314921745L;

        final Call call; // original call in source file
        final FPGARoutine parent;
        int index; // in parent

        StackOpCode(Call call, FPGARoutine parent) {
            super();
            this.call = call;
            this.parent = parent;
        }

        @Override
        public int getAddress() {
            return 0x300000 | (parent.getBaseAddress() + index);
        }
    }

    class StackFunctionOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = -1809130449104474507L;

        StackFunctionOpCode(Call call, FPGARoutine parent) {
            super(call, parent);
        }

        @Override
        public int getValue() {
            // 0x1: function exe + 4 bits function ID + 1 bit infinite loop + 23
            // bits repetition

            int i = functionsMap.get(call.getFunction());
            return 0x10000000 | (i << 24) | (call.isInfinity() ? (1 << 23) : 0) | (call.getRepeatValue());
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGFUNC;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = functionsMap.get(call.getFunction());
            cmnd[3] = call.getRepeatValue();
            return cmnd;
        }

        @Override
        public String getComment() {
            int i = functionsMap.get(call.getFunction());
            return String.format("function       %-20s #%02d    repeat %5d %s", call.getFunction().getId(), i,
                    call.getRepeatValue(), call.isInfinity() ? "infinity" : "");

            // return "function " + call.getFunction().getId() + " # " + i
            // + " repeat " + call.getRepeatValue() + " infinity "
            // + call.isInfinity();
        }
    }

    class StackFunctionPtrOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = -5100915003537149337L;

        StackFunctionPtrOpCode(Call call, FPGARoutine parent) {
            super(call, parent);
        }

        @Override
        public int getValue() {
            // 0x2: function exe + 4 bits function ID ptr + 1 bit infinite loop
            // + 23
            // bits repetition

            int i = functionsPtrMap.get(call.getFunctionPointer());
            return 0x20000000 | (i << 24) | (call.isInfinity() ? (1 << 23) : 0) | (call.getRepeatValue());
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGFUNC_FP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = functionsPtrMap.get(call.getFunctionPointer());
            cmnd[3] = call.getRepeatValue();
            return cmnd;
        }

        @Override
        public String getComment() {
            int i = functionsPtrMap.get(call.getFunctionPointer());
            return String.format("functionPtr    %-20s #%02d    repeat %5d %s", call.getFunctionPointer().getId(), i,
                    call.getRepeatValue(), call.isInfinity() ? "infinity" : "");
        }
    }

    class StackFunctionRepPtrOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = 3711825773613846285L;

        StackFunctionRepPtrOpCode(Call call, FPGARoutine parent) {
            super(call, parent);
        }

        @Override
        public int getValue() {
            // 0x3: function exe + 4 bits function ID + unused 20 bits
            // + 4 bits repeat and infinity pointer

            int i = functionsMap.get(call.getFunction());
            int j = functionsRepPtrMap.get(call.getRepeatFcnPtr());

            return 0x30000000 | (i << 24) | j;
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGFUNC_RP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = functionsMap.get(call.getFunction());
            cmnd[3] = functionsRepPtrMap.get(call.getRepeatFcnPtr());
            return cmnd;
        }

        @Override
        public String getComment() {
            int i = functionsMap.get(call.getFunction());
            int j = functionsRepPtrMap.get(call.getRepeatFcnPtr());
            return String.format("function       %-20s #%02d    repPtr %-20s #%02d %s", call.getFunction().getId(), i,
                    call.getRepeatFcnPtr().getId(), j, call.isInfinity() ? "infinity" : "");
        }
    }

    class StackFunctionPtrRepPtrOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = 1348240880246260161L;

        StackFunctionPtrRepPtrOpCode(Call call, FPGARoutine parent) {
            super(call, parent);
        }

        @Override
        public int getValue() {
            // 0x4: function exe + 4 bits function ID ptr + unused 20 bits
            // + 4 bits repeat and infinity pointer

            int i = functionsPtrMap.get(call.getFunctionPointer());
            int j = functionsRepPtrMap.get(call.getRepeatFcnPtr());

            return 0x40000000 | (i << 24) | j;
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGFUNC_FRP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = functionsPtrMap.get(call.getFunctionPointer());
            cmnd[3] = functionsRepPtrMap.get(call.getRepeatFcnPtr());
            return cmnd;
        }

        @Override
        public String getComment() {
            int i = functionsPtrMap.get(call.getFunctionPointer());
            int j = functionsRepPtrMap.get(call.getRepeatFcnPtr());
            return String.format("functionPtr    %-20s #%02d     repPtr %-20s #%02d %s",
                    call.getFunctionPointer().getId(), i, call.getRepeatFcnPtr().getId(), j,
                    call.isInfinity() ? "infinity" : "");
        }
    }

    static class StackSubroutineOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = -5570330889314120999L;

        FPGARoutine callee;

        StackSubroutineOpCode(Call call, FPGARoutine parent, FPGARoutine callee) {
            super(call, parent);
            this.callee = callee;
        }

        @Override
        public int getValue() {
            // return 0x20000000 | (callee.getBaseAddress() << 18)
            // | (call.isInfinity() ? (1 << 17) : 0)
            // | (call.getRepeatValue());
            return 0x50000000 | (callee.getBaseAddress() << 16) | (call.getRepeatValue());
            // 0x5: jump to address + 2 bits unused must be zero + 10 bits
            // address + 16 bits repetition
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGJUMP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = callee.getBaseAddress();
            cmnd[3] = call.getRepeatValue();
            return cmnd;
        }

        @Override
        public String getComment() {
            if (call.getSubroutine() != null) {
                return String.format("subroutine     %-20s #%05x repeat %5d", call.getSubroutine().getId(),
                        callee.getBaseAddress(), call.getRepeatValue());
            } else {
                return String.format("embedded       %-20s #%05x repeat %5d", callee.orgRoutine.getId(),
                        callee.getBaseAddress(), call.getRepeatValue());
            }
        }
    }

    class StackSubroutinePtrOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = -824618818863692467L;

        public StackSubroutinePtrOpCode(Call call, FPGARoutine parent) {
            super(call, parent);
        }

        @Override
        public int getValue() {
            int i = routinesPtrMap.get(call.getSubroutinePointer());
            return 0x60000000 | (i << 16) | (call.getRepeatValue());
            // 0x6: jump to address + 8 bits unused must be zero + 4 bits
            // address pointer + 16 bits repetition
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGJUMP_AP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = routinesPtrMap.get(call.getSubroutinePointer());
            cmnd[3] = call.getRepeatValue();
            return cmnd;
        }

        @Override
        public String getComment() {
            int i = routinesPtrMap.get(call.getSubroutinePointer());
            return String.format("subroutinePtr  %-20s #%02d     repeat %5d", call.getSubroutinePointer().getId(), i,
                    call.getRepeatValue());
        }
    }

    class StackSubroutineRepPtrOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = 3006672373753140538L;

        FPGARoutine callee;

        public StackSubroutineRepPtrOpCode(Call call, FPGARoutine parent, FPGARoutine callee) {
            super(call, parent);
            this.callee = callee;
        }

        @Override
        public int getValue() {
            int j = routinesRepPtrMap.get(call.getRepeatSubPtr());
            return 0x70000000 | (callee.getBaseAddress() << 16) | j;
            // 0x7: jump to address + 2 bits unused must be zero + 10 bits
            // address + 4 bits repetition pointer
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGJUMP_RP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = callee.getBaseAddress();
            cmnd[3] = routinesRepPtrMap.get(call.getRepeatSubPtr());
            return cmnd;
        }

        @Override
        public String getComment() {
            int j = routinesRepPtrMap.get(call.getRepeatSubPtr());
            return String.format("subroutine     %-20s #%05x repPtr %-20s #%02d", callee.orgRoutine.getId(),
                    callee.getBaseAddress(), call.getRepeatSubPtr().getId(), j);
        }
    }

    class StackSubroutinePtrRepPtrOpCode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = 1725715638050287585L;

        StackSubroutinePtrRepPtrOpCode(Call call, FPGARoutine parent) {
            super(call, parent);
        }

        @Override
        public int getValue() {
            int i = routinesPtrMap.get(call.getSubroutinePointer());
            int j = routinesRepPtrMap.get(call.getRepeatSubPtr());
            return 0x80000000 | (i << 16) | j;
            // 0x8: jump to address + 8 bits unused must be zero + 4 bits
            // address pointer + 12 unused bits + 4 bits repetition pointer
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[4];
            cmnd[0] = CMD_PROGJUMP_ARP;
            cmnd[1] = parent.getBaseAddress() + index;
            cmnd[2] = routinesPtrMap.get(call.getSubroutinePointer());
            cmnd[3] = routinesRepPtrMap.get(call.getRepeatSubPtr());
            return cmnd;
        }

        @Override
        public String getComment() {
            int i = routinesPtrMap.get(call.getSubroutinePointer());
            int j = routinesRepPtrMap.get(call.getRepeatSubPtr());
            return String.format("subroutine     %-20s #%02d     repPtr %-20s #%02d",
                    call.getSubroutinePointer().getId(), i, call.getRepeatSubPtr().getId(), j);
        }
    }

    static class StackReturnOpcode extends StackOpCode implements Serializable {

        private static final long serialVersionUID = 8623730968069893135L;
        private final boolean isMain;

        StackReturnOpcode(FPGARoutine parent) {
            super(null, parent);
            isMain = parent.orgRoutine instanceof JumpTable;
        }

        @Override
        public int getValue() {
            if (isMain) {
                return 0xf0000000;
            } else {
                return 0xe0000000;
            }
        }

        @Override
        public int[] getCommand() {
            int[] cmnd = new int[2];
            cmnd[0] = isMain ? CMD_PROGEND : CMD_PROGSUBE;
            cmnd[1] = parent.getBaseAddress() + index;
            return cmnd;
        }

        @Override
        public String getComment() {
            if (isMain) {
                return "return_main";
            } else {
                return "return_subr";
            }
        }
    }

    static class FPGARoutine implements Serializable {

        private static final long serialVersionUID = -268985060126274257L;

        Callable orgRoutine; // Subroutine, Main or EmbeddedSubroutine
        int baseAddress;
        final List<StackOpCode> opcodes = new ArrayList<>();

        public int getBaseAddress() {
            return baseAddress;
        }

        public void add(StackOpCode op) {
            op.index = opcodes.size();
            opcodes.add(op);
        }

        public void dump() {
            System.out.printf("// @ %3x : %s\n", baseAddress, orgRoutine.getId());
            opcodes.stream().forEach((op) -> {
                op.dump();
            });
        }
    }

    /**
     * Encapsulates information about a pointer.
     */
    public static class PointerInfo extends AddressAndValue implements Serializable {

        private static final long serialVersionUID = -8312653354624413268L;

        /**
         * The kind of pointer.
         */
        public static enum PointerKind {
            FUNCTION(Sequencer.REG_SEQ_EXEC_REFS), SUBROUTINE(Sequencer.REG_SEQ_JUMP_REFS), REPEAT_FUNCTION(
                    Sequencer.REG_SEQ_EXEC_REPS), REPEAT_SUBROUTINE(Sequencer.REG_SEQ_JUMP_REPS);

            private final int baseAddress;

            PointerKind(int baseAddress) {
                this.baseAddress = baseAddress;
            }

            private int getBaseAddress() {
                return baseAddress;
            }

        };

        private final PointerKind kind;
        private final String name;
        private final int offset;
        private final int value;

        PointerInfo(PointerKind kind, String name, int offset, int value) {
            super();
            this.kind = kind;
            this.name = name;
            this.offset = offset;
            this.value = value;
        }

        /**
         * The type of this pointer.
         *
         * @return
         */
        public PointerKind getKind() {
            return kind;
        }

        /**
         * The name of this pointer
         *
         * @return
         */
        public String getName() {
            return name;
        }

        /**
         * The offset of this pointer relative to its pointer block.
         *
         * @return
         */
        public int getOffset() {
            return offset;
        }

        @Override
        public int getAddress() {
            return offset + kind.getBaseAddress();
        }

        /**
         * The value of this pointer. Depending on the type of pointer this may
         * be an address or a repetition count.
         *
         * @return
         */
        @Override
        public int getValue() {
            return value;
        }

        @Override
        public String toString() {
            return String.format("PointerInfo{kind=%s, name=%s, address=%s, value=%s}", kind, name, offset, value);
        }

        public int[] getCommand() {
            int[] cmnd = new int[3];
            cmnd[0] = kind == PointerKind.FUNCTION ? CMD_POINTER_FA
                    : kind == PointerKind.REPEAT_FUNCTION ? CMD_POINTER_FR
                            : kind == PointerKind.SUBROUTINE ? CMD_POINTER_SA : CMD_POINTER_SR;
            cmnd[1] = offset;
            cmnd[2] = value;
            return cmnd;
        }
    }

    /**
     * Returns a list of pointer information, with kind, name and address.
     * Within each type the order in which the pointers appear corresponds to
     * the order in which they were declared in the source file.
     *
     * @return A list of pointer information, with kind, name, address and
     * value.
     */
    public List<PointerInfo> getPointers() {
        List<PointerInfo> l = new ArrayList<>();

        functionsPtrMap.forEach((k, v) -> l.add(new PointerInfo(PointerKind.FUNCTION, k.getId(),
                v, functionsMap.get(k.getFunction()))));
        routinesPtrMap.forEach((k, v) -> l.add(new PointerInfo(PointerKind.SUBROUTINE, k.getId(),
                v, routinesMap.get(k.getSubroutine()).getBaseAddress())));
        functionsRepPtrMap.forEach((k, v) -> l.add(
                new PointerInfo(PointerKind.REPEAT_FUNCTION, k.getId(), v, k.getN())));
        routinesRepPtrMap.forEach((k, v) -> l.add(
                new PointerInfo(PointerKind.REPEAT_SUBROUTINE, k.getId(), v, k.getN())));
        return l;
    }

    public Map<String, PointerInfo> getPointerMap() {
        LinkedHashMap<String, PointerInfo> result = new LinkedHashMap<>();
        functionsPtrMap.forEach((k, v) -> result.put(k.getId(), new PointerInfo(PointerKind.FUNCTION, k.getId(),
                v, functionsMap.get(k.getFunction()))));
        routinesPtrMap.forEach((k, v) -> result.put(k.getId(), new PointerInfo(PointerKind.SUBROUTINE, k.getId(),
                v, routinesMap.get(k.getSubroutine()).getBaseAddress())));
        functionsRepPtrMap.forEach((k, v) -> result.put(k.getId(),
                new PointerInfo(PointerKind.REPEAT_FUNCTION, k.getId(), v, k.getN())));
        routinesRepPtrMap.forEach((k, v) -> result.put(k.getId(),
                new PointerInfo(PointerKind.REPEAT_SUBROUTINE, k.getId(), v, k.getN())));
        return result;
    }

    /**
     * Return the list of addresses of all routines that are main.
     *
     * @return the map of name->addresses for all main routines. The order of
     * the elements in the list is the same as the order the elements are
     * declared in the file.
     */
    public Map<String, Integer> getMainAddresses() {

        Map<String, Integer> opcodes = getMainOpcodes();
        Map<String, Integer> m = new LinkedHashMap<>();
        routines.stream().filter((r) -> (r.orgRoutine instanceof Main)).forEach((r) -> {
            m.put(r.orgRoutine.getId(), opcodes.get(r.orgRoutine.getId())*4);
        });
        return m;
    }

    public Map<String, Integer> getMainOpcodes() {
        // The opcodes are a function of the order in which the mains get put into
        // the jump table, which is also the order in the routines list.
        int opcode = 0;
        Map<String, Integer> m = new LinkedHashMap<>();
        for (FPGARoutine routine : routines) {
            if (routine.orgRoutine instanceof Main) {
                m.put(routine.orgRoutine.getId(), opcode++);
            }
        }
        return m;
    }

    /**
     * Return the list of addresses of all subroutines.
     *
     * @return the map of name->addresses for all subroutines. The order of the
     * elements in the map returned is the order of the elements in the input
     * file.
     */
    public Map<String, Integer> getSubroutineAddresses() {
        Map<String, Integer> m = new LinkedHashMap<>();
        routines.stream().filter((r) -> (!(r.orgRoutine instanceof Main))).filter((r) -> !"JumpTable".equals(r.orgRoutine.getId())).forEach((r) -> {
            m.put(r.orgRoutine.getId(), r.getBaseAddress());
        });
        return m;
    }

    /**
     * Return a map of all functions.
     *
     * @return the map of name->ordinal for all functions. The order of the
     * functions within the map is the order they were declared in the input
     * file.
     */
    public Map<String, Integer> getFunctionAddresses() {
        Map<String, Integer> m = new LinkedHashMap<>();
        functionsMap.keySet().stream().forEach((f) -> {
            m.put(f.getId(), functionsMap.get(f));
        });
        return m;
    }

    /**
     * Return the metadata (a.k.a. constants or parameters) associated with this
     * model
     *
     * @return An unmodifiable map of the metadata, in the form name->value. The
     * order of the elements in the map is the order they appear in the input
     * file. The value is as it appears in the input file (e.g. 2000 ns).
     */
    public Map<String, String> getMetadata() {
        return Collections.unmodifiableMap(metadata);
    }

    /**
     * Return the channels (a.k.a clocks) associated with this model
     *
     * @return The map of channels, in the form name->value. The order of the
     * elements in the map is the order they appeared in the input file.
     */
    public Map<String, Integer> getChannels() {
        return Collections.unmodifiableMap(channels);
    }

    void addMetadata(String key, String value) {
        metadata.put(key, value);
    }

    void addChannel(String id, int value) {
        channels.put(id, value);
    }

    public Set<String> getMetadataKeys() {
        return metadata.keySet();
    }

    public String getMetadata(String key) {
        return metadata.get(key);
    }

    public void dump() {
        timing.keySet().stream().forEach((addr) -> {
            timing.get(addr).dump();
        });

        routines.stream().forEach((r) -> {
            r.dump();
        });

        System.out.printf("00400000 %08x // last address\n", lastAddr);

        System.out.println("---");
        System.out.println("--- POINTERS");
        System.out.println("---");

        getPointers().stream().forEach((i) -> {
            System.out.printf("%-20s  %03x  %s\n", i.getKind(), i.getOffset(), i.getName());
        });

        System.out.println("---");
        System.out.println("--- MAINS");
        System.out.println("---");

        getMainAddresses().entrySet().stream().forEach((i) -> {
            System.out.printf("%-20s  %08x \n", i.getKey(), i.getValue());
        });
    }

    /**
     * Returns the complete register memory map for this model. Note that this
     * method is not currently used by the rafts subsystem, but can be used for
     * loading the sequencer simulator for generating waveforms from the model.
     *
     * @return A complete list of AddressAndValue objects representing the
     * memory map.
     */
    public List<AddressAndValue> getMemoryMap() {
        List<AddressAndValue> result = new ArrayList<>();

        // All function times and output values
        result.addAll(timing.values());

        // Routines include all subroutines, and mains
        for (FPGARoutine r : routines) {
            result.addAll(r.opcodes);
        }

        result.addAll(getPointers());

        return result;
    }

    /**
     * Compute a checksum for the compiled model
     *
     * @return The computed checksum
     */
    public long computeCheckSum() {
        List<AddressAndValue> memoryMap = getMemoryMap();
        Checksum cksum = new CRC32();
        ByteBuffer bb = ByteBuffer.allocate(8);
        bb.order(ByteOrder.BIG_ENDIAN);
        for (AddressAndValue av : memoryMap) {
            bb.clear();
            bb.putInt(av.getAddress());
            bb.putInt(av.getValue());
            cksum.update(bb.array(), 0, 8);
        }
        return cksum.getValue();
    }

    /**
     * Dumps out the model in the format required by the rafts subsystem.
     *
     * @return
     */
    public List<int[]> getCommands() {
        List<int[]> commands = new ArrayList<>();
        timing.values().stream().forEach((r) -> {
            commands.add(r.getCommand());
        });
        routines.stream().forEach((r) -> {
            r.opcodes.stream().forEach((c) -> {
                commands.add(c.getCommand());
            });
        });
        getPointers().stream().forEach((p) -> {
            commands.add(p.getCommand());
        });

        return commands;
    }

    public void validate() {
        // Restrict indirect parameter names so that a given name can occur only
        // once, regardless of what type (function pointer, function count
        // pointer, subroutine pointer or subroutine count pointer) it is.

        List<PointerInfo> l = getPointers();
        Set<String> names = new HashSet<>();
        for (PointerInfo p : l) {
            if (names.contains(p.getName())) {
                throw new RuntimeException("duplicate pointer name " + p.getName());
            }
            names.add(p.getName());
        }

        // Don't allow more than 16 (used) pointer definitions of each type.
        // Don't allow a function to contain only one time slice - it renders
        // the sequencer inoperable until the firmware is reset.
        for (Function f : functionsMap.keySet()) {
            if (f.getId().equals("Default")) {
                continue;
            }
            if (f.getTimeslices().size() <= 1) {
                throw new RuntimeException("Function too short " + f.getId() + " " + f.getTimeslices().size());
            }
        }
    }
}
