package org.lsst.ccs.subsystem.rafts.config;

import java.io.Serializable;
import org.lsst.ccs.drivers.reb.Asic;

/**
 * REB configuration. One REB manages 3 CCD. A raft is handled by 3 REB.
 * 
 * @author aubourg
 * 
 *         One REB = 3 CCD. One raft = 3 REB.
 * 
 *         One REB corresponds to one FPGA.
 * 
 *         One REB corresponds to up to 6 ASPIC, since one ASPIC manages half a CCD.
 * 
 *         In a given configuration, all REBs share the same register mapping,
 *         each has an id.
 */
public class REB implements Serializable {

    // This could belong to a raft... 3 REB per raft.

    public static final int
        N_CCD   = Asic.MAX_STRIPS,
        N_ASIC  = Asic.MAX_ASICS;

    public static final int VERSION_UNSUPP = -1;
    public static final int VERSION_0 = 0;
    public static final int VERSION_1 = 1;
    public static final int VERSION_2 = 2;
    public static final int VERSION_3 = 3;

    private int
        dacVersion,
        biasVersion,
        aspicVersion;  // Component versions
    private boolean dacRaw, biasRaw;  // Raw data state of components

    private int id; // this is the id used to communicate with the REB
    private String ifcName;  // The network interface used for the REB

    private int ccdMask = 1;    // mask of CCDs we are driving: 1-7
    private int maxCcdMask = 7; // maximal mask of CCDs we can drive

    private final DACS dacs = new DACS();
    private final BiasDACS[] biases = new BiasDACS[N_CCD];
    private final ASPIC[] aspics = new ASPIC[N_ASIC];

    public int getDacVersion() {
        return dacVersion;
    }

    public void setDacVersion(int version) {
        this.dacVersion = version;
    }

    public int getBiasVersion() {
        return biasVersion;
    }

    public void setBiasVersion(int version) {
        this.biasVersion = version;
    }

    public int getAspicVersion() {
        return aspicVersion;
    }

    public void setAspicVersion(int version) {
        this.aspicVersion = version;
    }

    public boolean isDacRaw() {
        return dacRaw;
    }

    public void setDacRaw(boolean state) {
        this.dacRaw = state;
    }

    public boolean isBiasRaw() {
        return biasRaw;
    }

    public void setBiasRaw(boolean state) {
        this.biasRaw = state;
    }

    public int getId() {
        return id;
    }

    public void setId(int id) {
        this.id = id;
    }

    public String getIfcName() {
        return ifcName;
    }

    public void setIfcName(String ifcName) {
        this.ifcName = ifcName;
    }

    public int getCcdMask() {
        return ccdMask;
    }

    public void setCcdMask(int ccdMask) {
        if ((ccdMask & maxCcdMask) != ccdMask) {
            throw new IllegalArgumentException("ccdMask (" + ccdMask
                                                 + ") should be submask of "
                                                 + maxCcdMask);
        }
        this.ccdMask = ccdMask;
    }

    public int getMaxCcdMask() {
        return maxCcdMask;
    }

    public void setMaxCcdMask(int maxCcdMask) {
        this.maxCcdMask = maxCcdMask;
    }

    public DACS getDacs() {
        return dacs;
    }

    public BiasDACS[] getBiases() {
        return biases;
    }

    public ASPIC[] getAspics() {
        return aspics;
    }

    public void copyFrom(REB reb) {
        id = reb.id;
        ifcName = reb.ifcName;
        ccdMask = reb.ccdMask;
        maxCcdMask = reb.maxCcdMask;
        dacVersion = reb.dacVersion;
        dacs.copyFrom(reb.dacs);
        biasVersion = reb.biasVersion;
        for (int j = 0; j < N_CCD; j++) {
            BiasDACS bias = biases[j];
            if (bias != null) {
                bias.copyFrom(reb.biases[j]);
            }
        }
        aspicVersion = reb.aspicVersion;
        for (int j = 0; j < N_ASIC; j++) {
            ASPIC aspic = aspics[j];
            if (aspic != null) {
                aspic.copyFrom(reb.aspics[j]);
            }
        }
    }

    private static final long serialVersionUID = -630477405631882175L;
}
