package org.lsst.ccs.subsystem.rafts.fpga.compiler;

import java.util.List;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.EndSliceTime;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.FPGARoutine;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.SliceTime;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.SliceValues;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.StackOpCode;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.StackReturnOpcode;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.StackSubroutineOpCode;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.StackSubroutineRepPtrOpCode;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Call;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Callable;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Channel;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Function;
import org.lsst.ccs.subsystem.rafts.fpga.xml.FunctionPointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Main;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Parameter;
import org.lsst.ccs.subsystem.rafts.fpga.xml.RepeatFunctionPointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.RepeatSubroutinePointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Sequencer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Subroutine;
import org.lsst.ccs.subsystem.rafts.fpga.xml.SubroutinePointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Timeslice;

/**
 * The visitor which builds the final FPGA2Model
 * @author aubourg
 */
class FPGA2ModelBuilderVisitor extends ModelBuilderVisitor {
    
    private final FPGA2Model model = new FPGA2Model();
    private int nMains = 0;
    int curFuncIndex = 0;
    int nextFuncIndex = 1;
    int curFuncPtrIndex = 0;
    
    public FPGA2Model getModel() {
        return model;
    }

    @Override
    public void visit(Function f) {
        if (f.getId().equals("Default")) {
            curFuncIndex = 0;
        } else {
            curFuncIndex = nextFuncIndex;
            nextFuncIndex++;
        }
        model.functionsMap.put(f, curFuncIndex);
        super.visit(f);
        if (sliceIndex < 16) {
            EndSliceTime t = new EndSliceTime(curFuncIndex, f, sliceIndex);
            model.timing.put(t.getAddress(), t);
        }
    }

    @Override
    public void visit(Timeslice s) {
        super.visit(s);
        SliceValues v = new SliceValues(curFuncIndex, (Function) current, s);
        model.timing.put(v.getAddress(), v);
        SliceTime t = new SliceTime(curFuncIndex, (Function) current, s);
        model.timing.put(t.getAddress(), t);
    }

    @Override
    public void visit(Subroutine s) {
        super.visit(s);
        FPGARoutine r = buildRoutine(s);
        model.routines.add(r);
        model.routinesMap.put(s, r);
    }

    @Override
    public void visit(Main m) {
        super.visit(m);

        // We put the mains at the beginning of the list of routines
        // but in the order they are found in the file.
        FPGARoutine r = buildRoutine(m);
        model.routines.add(nMains++, r); 
        model.routinesMap.put(m, r);
    }

    @Override
    public void visit(FunctionPointer fp) {
        model.functionsPtrMap.put(fp, curFuncPtrIndex);
        curFuncPtrIndex++;
        super.visit(fp);
    }
    int curSubPtrIndex = 0;

    @Override
    public void visit(SubroutinePointer sp) {
        model.routinesPtrMap.put(sp, curSubPtrIndex);
        curSubPtrIndex++;
        super.visit(sp);
    }
    int curRepFuncPtrIndex = 0;

    @Override
    public void visit(RepeatFunctionPointer rfp) {
        model.functionsRepPtrMap.put(rfp, curRepFuncPtrIndex);
        curRepFuncPtrIndex++;
        super.visit(rfp);
    }
    int curRepSubPtrIndex = 0;

    @Override
    public void visit(RepeatSubroutinePointer rsp) {
        model.routinesRepPtrMap.put(rsp, curRepSubPtrIndex);
        curRepSubPtrIndex++;
        super.visit(rsp);
    }

    private FPGARoutine buildRoutine(Callable c) {
        FPGARoutine r = new FPGARoutine();
        r.orgRoutine = c;
        List<Call> calls;
        if (c instanceof Subroutine) {
            calls = ((Subroutine) c).getCalls();
        } else if (c instanceof EmbeddedSubroutine) {
            calls = ((EmbeddedSubroutine) c).getCalls();
        } else {
            throw new RuntimeException("type mismatch for " + c + " should be Subroutine or Embedded");
        }
        calls.stream().forEach((org.lsst.ccs.subsystem.rafts.fpga.xml.Call call) -> {
            if (call.getFunction() != null) {
                if (call.getRepeatFcnPtr() == null) {
                    StackOpCode op = model.new StackFunctionOpCode(call, r);
                    r.add(op);
                } else {
                    StackOpCode op = model.new StackFunctionRepPtrOpCode(call, r);
                    r.add(op);
                }
            } else if (call.getFunctionPointer() != null) {
                if (call.getRepeatFcnPtr() == null) {
                    StackOpCode op = model.new StackFunctionPtrOpCode(call, r);
                    r.add(op);
                } else {
                    StackOpCode op = model.new StackFunctionPtrRepPtrOpCode(call, r);
                    r.add(op);
                }
            } else if (call.getSubroutine() != null) {
                if (call.getRepeatSubPtr() == null) {
                    StackOpCode op = new StackSubroutineOpCode(call, r, null);
                    // we will find later the correct FPGARoutine to insert
                    r.add(op);
                } else {
                    StackOpCode op = model.new StackSubroutineRepPtrOpCode(call, r, null);
                    // we will find later the correct FPGARoutine to insert
                    r.add(op);
                }
            } else if (call.getSubroutinePointer() != null) {
                if (call.getRepeatSubPtr() == null) {
                    StackOpCode op = model.new StackSubroutinePtrOpCode(call, r);
                    r.add(op);
                } else {
                    StackOpCode op = model.new StackSubroutinePtrRepPtrOpCode(call, r);
                    r.add(op);
                }
            } else {
                Subroutine parent;
                if (c instanceof Subroutine) {
                    parent = (Subroutine) c;
                } else {
                    parent = ((EmbeddedSubroutine) c).getParent();
                }
                EmbeddedSubroutine es = new EmbeddedSubroutine(parent);
                es.setCalls(call.getCalls());
                FPGARoutine er = buildRoutine(es);
                model.routines.add(er);
                if (call.getRepeatSubPtr() == null) {
                    StackOpCode op = new StackSubroutineOpCode(call, r, er);
                    r.add(op);
                } else {
                    StackOpCode op = model.new StackSubroutineRepPtrOpCode(call, r, er);
                    r.add(op);
                }
            }
        });
        r.add(new StackReturnOpcode(r));
        return r;
    }

    private void completeRoutines() {
        // first, find all missing FPGARoutines
        model.routines.stream().forEach((FPGARoutine r) -> {
            r.opcodes.stream().forEach((StackOpCode op) -> {
                if (op instanceof StackSubroutineOpCode) {
                    StackSubroutineOpCode sop = (StackSubroutineOpCode) op;
                    if (sop.callee == null) {
                        FPGARoutine rr = model.routinesMap.get(sop.call.getSubroutine());
                        sop.callee = rr;
                    }
                } else if (op instanceof StackSubroutineRepPtrOpCode) {
                    StackSubroutineRepPtrOpCode sop = (StackSubroutineRepPtrOpCode) op;
                    if (sop.callee == null) {
                        FPGARoutine rr = model.routinesMap.get(sop.call.getSubroutine());
                        sop.callee = rr;
                    }
                }
            });
        });
        int addr = 0;
        for (FPGARoutine r : model.routines) {
            r.baseAddress = addr;
            addr += r.opcodes.size();
            // round up : THIS IS AN LPNHE REQUEST. They like 0-ending hex addresses.
            // please keep this.
            addr = ((addr + 16) / 16) * 16;
        }
        model.lastAddr = addr - 1;
    }

    @Override
    public void visit(Sequencer s) {
        super.visit(s);
        completeRoutines();
        updateMetadata(s);
        updateChannels(s);
    }

    private void updateMetadata(Sequencer s) {
        List<Parameter> params = s.getSequencerConfig().getParameters();
        params.stream().forEach((p) -> {
            model.addMetadata(p.getId(), p.getValue());
        });
    }

    private void updateChannels(Sequencer s) {
        List<Channel> channels = s.getSequencerConfig().getChannels();
        channels.stream().forEach((c) -> {
            model.addChannel(c.getId(), c.getValue());
        });
    }
    
}
