package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.subsystem.common.actions.BulkPsPowerAction;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.power.data.PowerDataGroup;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.PowerSupplyState;
import org.lsst.ccs.subsystem.power.states.RebDPhiState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 *  Implements the REB PS control subsystem.
 *
 *  @author The LSST CCS Team
 */
public class RebPowerSupplyMain extends Subsystem implements HasLifecycle, HasDataProviderInfos {

    private static final Logger LOG = Logger.getLogger(RebPowerSupplyMain.class.getName());

    @LookupField( strategy=LookupField.Strategy.CHILDREN)
    private List<PowerDevice> mainPsList = new ArrayList<>();
    @LookupField( strategy=LookupField.Strategy.CHILDREN)    
    private List<RebPsDevice> psDeviceList = new ArrayList<>();

        
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dictionaryService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private EmergencyResponseManager emergencyResponseManager;
    
    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String,RebPowerSupplyNode> rebs = new HashMap<>();
    
    @ConfigurationParameter
    private volatile long powerOffSleepMillis = 100;

    @ConfigurationParameter(name="psDeviceReadTimeout", description = "Read Timeout for the underlying PS driver", units = "ms")
    private volatile long psDeviceReadTimeout = 1000;    
    
    @ConfigurationParameter(category = "General", description = "Number of consecutive read failures (exceptions) before gonig Offline. Must be a positive number greater than zero.", range = "1..5", units = "unitless")    
    private volatile int psDeviceExcepNumToOffline = 1;
    
    //The map of all the Channels
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String,Channel> channels = new HashMap<>();
    
    public RebPowerSupplyMain() {
        super("rebps", AgentInfo.AgentType.WORKER);
    }

    
    @Override
    public void build() {
        if (emergencyResponseManager == null) {
            throw new RuntimeException("Cannot operate the RebPower subsystem without an EmergencyResponseManager");
        }
        
        agentPropertiesService.setAgentProperty(BulkPsPowerAction.class.getCanonicalName(), "uses");
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.POWER.name());
        agentPropertiesService.setAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, HasFocalPlaneData.generatePropertyValue(PowerDataGroup.class));
        AgentPeriodicTask pt;
        //Create and schedule an AgentPeriodicTask to update the RebPowerState
        pt = new AgentPeriodicTask("reb-power-state",
                                   () -> updateRebPowerState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }
    
    @Override
    public void init()
    {
        for (RebPowerSupplyNode reb : rebs.values()) {
            agentStateService.registerState(RebPowerState.class, "REB power state", reb);        
            agentStateService.registerState(RebHvBiasState.class, "REB HV bias state", reb);        
            if ( reb instanceof CornerRaftRebPowerSupplyNode ) {
                agentStateService.registerState(RebDPhiState.class, "REB Dphi state", reb);        
            }
        
        }        
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            agentStateService.registerState(PowerSupplyState.class, "Power Supply state", powerSupplyDevice);
        }
        
        setPsDeviceReadTimeout(psDeviceReadTimeout);
        setPsDeviceConsecutiveExceptionsToOffline(psDeviceExcepNumToOffline);
    }
    
    @ConfigurationParameterChanger(propertyName = "psDeviceExcepNumToOffline") 
    public void setPsDeviceConsecutiveExceptionsToOffline(int value) {
        psDeviceExcepNumToOffline = value;
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            powerSupplyDevice.setConsecutiveExceptiionsToOffline(psDeviceExcepNumToOffline);
        }
        
    }

    @ConfigurationParameterChanger(propertyName = "psDeviceReadTimeout") 
    public void setPsDeviceReadTimeout(long timeout) {
        if ( timeout <= 0 ) {
            throw new IllegalArgumentException("Negative timeouts are not allowd");
        }
        psDeviceReadTimeout = timeout;
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            powerSupplyDevice.setReadTimeout(psDeviceReadTimeout);
        }
        
    }
    
    @Override
    public void finalizeDictionary() {        
        //Manipulate the dictionary here: 
        for ( DataProviderInfo data: dictionaryService.getDataProviderDictionary().getDataProviderInfos() ) {
            PowerDataGroup dataGroup = PowerDataGroup.findPowerDataGroup(data);
            if (dataGroup != null) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }
    
    

    /**
     *  Updates the REB power state periodically.
     */
    private void updateRebPowerState() {
        StateBundle sb = new StateBundle();        
        
        //Update the Power Supply state
        for (RebPsDevice powerSupplyDevice : psDeviceList ) {
            StateBundle psDeviceState = powerSupplyDevice.getRebPsDeviceState();
            sb = sb.mergeState(psDeviceState);
        }        
        agentStateService.updateAgentState(sb);
    }


    @Command
    public void powerOffAllRebs() throws InterruptedException {
        String failedRebs = "";
        int count = rebs.size();
        for (RebPowerSupplyNode node: rebs.values()) {
            count--;
            if ( agentStateService.isComponentInState(node.rebPath, RebPowerState.ON) ) {
                LOG.log(Level.INFO, "Powering off reb: {0}", new Object[]{node.rebPath});
                try {
                    node.powerRebOff();
                } catch (PowerException ex) {
                    LOG.log(Level.WARNING, "Failed to power off reb: "+node.rebPath, ex);                    
                    failedRebs += node.rebPath + " ";
                }
                if ( count > 0 ) {
                    Thread.sleep(powerOffSleepMillis);
                }
            }
        }
        if ( ! failedRebs.isEmpty() ) {
            throw new RuntimeException("Failed to turn off the following rebs: "+failedRebs.trim());
        }
    }
        
}
