package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Interface to an ASCII-commanded power supply device.
 *
 *  @author Owen Saxton
 */
public class PowerDeviceGroup {

    /**
     *  Constants
     */
    private static final Map<DriverConstants.ConnType, Power.ConnType> cTypeMap = new HashMap();
    static {
        cTypeMap.put(DriverConstants.ConnType.NET, Power.ConnType.Network);
        cTypeMap.put(DriverConstants.ConnType.SERIAL, Power.ConnType.Serial);
    }

    /**
     *  Data fields.
     */
    @LookupField(strategy=LookupField.Strategy.SIBLINGS)
    private final Map<String, PowerControl> controlsMap = new LinkedHashMap<>();   // Control channels
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private PowerDevice device;


    /**
     *  Gets the configuration data.
     *
     *  @return  A power configuration object
     */
    public Power getPowerConfig()
    {
        Power power = new Power();
        power.setName(device.getName());
        power.setConnType(cTypeMap.get(device.getConnType()));
        power.setDevcId(device.getDevcId());
        power.setDevcParm(device.getDevcParm());
        Power.Channel[] chans = new Power.Channel[controlsMap.size()];
        int j = 0;
        for (PowerControl ctl : controlsMap.values()) {
            chans[j++] = ctl.getConfig();
        }
        power.setChannels(chans);

        return power;
    }


    /**
     *  Sets the configuration data.
     *
     *  @param  power  A power configuration object
     *  @throws  PowerException
     */
    public void setPowerConfig(Power power) throws PowerException
    {
        for (Power.Channel chan : power.getChannels()) {
            String cName = device.getName() + "." + chan.getName();
            (controlsMap.get(cName)).setConfig(chan);
        }
    }


    /**
     *  Gets the list of power state data.
     *
     *  @return  A list of power state data
     */
    protected List<PowerChanState> getPowerState()
    {
        device.readVoltageGroup();
        device.readCurrentGroup();
        device.readOutputGroup();
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerControl ctl : controlsMap.values()) {
            pState.add(ctl.getState());
        }
        return pState;
    }


    /**
     *  Powers on all channels.
     *
     *  @throws  PowerException
     */
    protected void powerOn() throws PowerException
    {
        for (PowerControl ctl : controlsMap.values()) {
            ctl.writeAll();
        }
        try {
            device.writeVoltageGroup();
            device.writeCurrentGroup();
            device.writeOnDelayGroup();
            device.writeOffDelayGroup();
        }
        catch (DriverException e) {
            device.handleException(e);
        }
        setOutput(true);
    }


    /**
     *  Waits for power-on to complete.
     *
     *  @param  timeout  The timeout period (ms)
     *  @throws  PowerException
     */
    protected void waitPowerOn(int timeout) throws PowerException
    {
        waitOutputSet(true, timeout);
    }


    /**
     *  Powers off all channels.
     *
     *  @throws  PowerException
     */
    protected void powerOff() throws PowerException
    {
        setOutput(false);
    }


    /**
     *  Waits for power-off to complete.
     *
     *  @param  timeout  The timeout period (ms)
     *  @throws  PowerException
     */
    protected void waitPowerOff(int timeout) throws PowerException
    {
        waitOutputSet(false, timeout);
    }


    /**
     *  Sets the power output state for all channels.
     *
     *  @param  value  The output state to set
     *  @throws  PowerException
     */
    private void setOutput(boolean value) throws PowerException
    {
        try {
            for (PowerControl ctl : controlsMap.values()) {
                device.writeOutput(value, ctl.getHwChan());
            }
            device.writeOutputGroup();
        }
        catch (DriverException e) {
            device.handleException(e);
        }
    }


    /**
     *  Waits for the power output state to be set for all channels.
     *
     *  @param  value  The output state that was set
     *  @param  timeout  The maximum time to wait (ms)
     *  @throws  PowerException
     */
    private void waitOutputSet(boolean value, int timeout) throws PowerException
    {
        if (timeout <= 0) return;
        long maxTime = System.currentTimeMillis() + timeout;
        try {
            device.readOutputGroup();
            for (PowerControl ctl : controlsMap.values()) {
                while (System.currentTimeMillis() <= maxTime) {
                    if (value == device.readOutput(ctl.getHwChan())) return;
                    try {
                        Thread.sleep(10);
                    }
                    catch (InterruptedException e) {}
                }
            }
            throw new DriverTimeoutException("Output wait timeout");
        }
        catch (DriverException e) {
            device.handleException(e);
        }
    }

}
