package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentPropertyPredicate;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.power.constants.ATSChannels;
import org.lsst.ccs.subsystem.power.constants.MonitorControl;
import org.lsst.ccs.subsystem.power.data.ATSPowerState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;

/**
 *  Implements the Auxiliary Telescope power control subsystem.
 *
 *  @author Owen Saxton
 */
public class ATSPowerMain extends Subsystem implements HasLifecycle {

    private static final String
        DIGITAL_CHAN = "Digital",
        ANALOG_CHAN = "Analog",
        CLK_HIGH_CHAN = "ClockHigh",
        CLK_LOW_CHAN = "ClockLow",
        OD_CHAN = "OD",
        HV_BIAS_CHAN = "HVBias",
        DPHI_CHAN = "DPHI",
        OTM_CHAN = "OTM",
        FAN_CHAN = "Fan",
        AUX_CHAN = "Aux";

    private static final String[] pwrChannels = new String[ATSChannels.NUM_CHANNELS];
    static {
        pwrChannels[ATSChannels.CHAN_DIGITAL] = DIGITAL_CHAN;
        pwrChannels[ATSChannels.CHAN_ANALOG] = ANALOG_CHAN;
        pwrChannels[ATSChannels.CHAN_CLK_HIGH] = CLK_HIGH_CHAN;
        pwrChannels[ATSChannels.CHAN_CLK_LOW] = CLK_LOW_CHAN;
        pwrChannels[ATSChannels.CHAN_OD] = OD_CHAN;
        pwrChannels[ATSChannels.CHAN_DPHI] = DPHI_CHAN;
        pwrChannels[ATSChannels.CHAN_HV_BIAS] = HV_BIAS_CHAN;
        pwrChannels[ATSChannels.CHAN_OTM] = OTM_CHAN;
        pwrChannels[ATSChannels.CHAN_FAN] = FAN_CHAN;
        pwrChannels[ATSChannels.CHAN_AUX] = AUX_CHAN;
    }
    private static final int POWER_TIMEOUT = 2000;  // Power on/off timeout per channel
    private static final int FAN_CONTROL_TIME = 5000;    // Fan control period
    private static final int STATE_UPDATE_TIME = 3000;    // Power state update period
    private static final int[] rebChannels = {ATSChannels.CHAN_DIGITAL, ATSChannels.CHAN_ANALOG, ATSChannels.CHAN_OD,
                                              ATSChannels.CHAN_CLK_HIGH, ATSChannels.CHAN_CLK_LOW};

    private static final Logger LOG = Logger.getLogger(ATSPowerMain.class.getName());
    
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService agentAggregatorService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private PersistencyService persistencyService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, PowerControl> specPwrControls = new LinkedHashMap<>();

    @ConfigurationParameter (description = "Temperature limit", units = "C")
    private volatile double tempLimit;
    @ConfigurationParameter (description = "Temperature limit dead zone", units = "C")
    private volatile double tempDeadZone;

    private String tempChannel;

    @Persist
    private boolean tempFanControl;

    private PowerControl dphiControl, hvBiasControl, otmControl, fanControl, auxControl;
    private final Boolean[] powerOn = new Boolean[ATSChannels.NUM_CHANNELS];
    private final PowerGroup rebGroup = new PowerGroup(), dphiGroup = new PowerGroup(), hvBiasGroup = new PowerGroup(),
                             otmGroup = new PowerGroup(), fanGroup = new PowerGroup(), auxGroup = new PowerGroup();
    private final Map<String, PowerControl> pwrControls = new LinkedHashMap<>();
    private MonitorTaskControl monitorControl;
    double tempValue;


    public ATSPowerMain() {
        super("ats-power", AgentInfo.AgentType.WORKER);
    }
    
    
    /**
     *  Power subsystem initialization
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to automatically control the fan
        AgentPeriodicTask pt = new AgentPeriodicTask("fan-control",
                                                     () -> controlFan()).withPeriod(Duration.ofMillis(FAN_CONTROL_TIME));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Create and schedule an AgentPeriodicTask to update and publish the ATS power system state
        pt = new AgentPeriodicTask("power-state", () -> updatePowerState()).withPeriod(Duration.ofMillis(STATE_UPDATE_TIME));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Configure persistency service
        persistencyService.setAutomatic(true, true);
        persistencyService.setPersistencyFileName("persistence_" + name + ".properties");
    }


    /**
     *  Power subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a Power Supply controller.
        agentPropertiesService.setAgentProperty(PowerAgentProperties.ATS_POWER_AGENT, getClass().getCanonicalName());

        // Remove path prefix from power control channel names
        for (Map.Entry<String, PowerControl> e : specPwrControls.entrySet()) {
            String[] fields = e.getKey().split("/");
            pwrControls.put(fields[fields.length - 1], e.getValue());
        }

        // Checks power control channels
        boolean error = false;
        for (String chan : pwrChannels) {
            if (pwrControls.get(chan) == null) {
                LOG.log(Level.SEVERE, "Required power channel {0} has not been defined", chan);
                error = true;
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }

        dphiControl = pwrControls.get(DPHI_CHAN);
        hvBiasControl = pwrControls.get(HV_BIAS_CHAN);
        otmControl = pwrControls.get(OTM_CHAN);
        fanControl = pwrControls.get(FAN_CHAN);
        auxControl = pwrControls.get(AUX_CHAN);
        for (int chan : rebChannels) {
            rebGroup.addControl(pwrControls.get(pwrChannels[chan]));
        }
        dphiGroup.addControl(dphiControl);
        hvBiasGroup.addControl(hvBiasControl);
        otmGroup.addControl(otmControl);
        fanGroup.addControl(fanControl);
        auxGroup.addControl(auxControl);

        // Establish temperature channel for fan control
        if (tempChannel == null) {
            LOG.log(Level.SEVERE, "Required field tempChannel has not been defined");
            throw new RuntimeException("Fatal initialization error");
        }
        String[] tempTokens = tempChannel.split("/", 2);
        if (tempTokens.length < 2) {
            LOG.log(Level.SEVERE, "Field tempChannel has no subsystem name");
            throw new RuntimeException("Fatal initialization error");
        }
        AgentPropertyPredicate app = new AgentPropertyPredicate("[agentName:" + tempTokens[0] + "]");
        agentAggregatorService.setAggregatePattern(app, tempTokens[1]);
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart()
    {
        LOG.info("ATS power subsystem started");
    }


    /**
     *  Turns on the REB power, except for HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws PowerException
    {
        try {
            if (!isPowerOn()) {
                //System.out.println("Power on start time = " + System.currentTimeMillis());
                rebGroup.powerOn();
                rebGroup.waitPowerOn(POWER_TIMEOUT);
                //System.out.println("Power on end time = " + System.currentTimeMillis());
            }
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off all the REB power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws PowerException
    {
        PowerException excp = null;
        //System.out.println("Power off start time = " + System.currentTimeMillis());
        try {
            hvBiasGroup.powerOff();
        }
        catch (PowerException e) {
            excp = e;
        }
        try {
            dphiGroup.powerOff();
        }
        catch (PowerException e) {
            excp = excp == null ? e : excp;
        }
        try {
            rebGroup.powerOff();
            rebGroup.waitPowerOff(POWER_TIMEOUT);
            if (excp != null) {
                throw excp;
            }
        }
        finally {
        //System.out.println("Power off end time = " + System.currentTimeMillis());
            updatePowerState();
        }
    }


    /**
     *  Turns on the DPHI voltage.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the DPHI voltage")
    public void dphiOn() throws PowerException
    {
        if (!isPowerOn()) return;
        try {
            dphiGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the DPHI voltage.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the DPHI voltage")
    public void dphiOff() throws PowerException
    {
        try {
            dphiGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Queries the DPHI voltage state.
     *
     *  @return  Whether the DPHI voltage is on
     */
    @Command(type=CommandType.QUERY, description="Query the DPHI voltage state", level=0)
    public boolean isDphiOn()
    {
        return powerOn[ATSChannels.CHAN_DPHI] == Boolean.TRUE;
    }


    /**
     *  Sets the DPHI voltage
     *
     *  @param  value  The voltage to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set the DPHI voltage")
    public void setDphi(@Argument(description="Voltage value") double value) throws PowerException
    {
        dphiControl.setVoltage(value);
        try {
            if (powerOn[ATSChannels.CHAN_DPHI] == Boolean.TRUE) {
                dphiControl.writeVoltage();
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns on the HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the HV bias")
    public void hvBiasOn() throws PowerException
    {
        if (!isPowerOn()) return;
        try {
            hvBiasGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the HV bias.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the HV bias")
    public void hvBiasOff() throws PowerException
    {
        try {
            hvBiasGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Queries the HV bias voltage state.
     *
     *  @return  Whether the HV bias voltage is on
     */
    @Command(type=CommandType.QUERY, description="Query the HV bias voltage state", level=0)
    public boolean isHvBiasOn()
    {
        return powerOn[ATSChannels.CHAN_HV_BIAS] == Boolean.TRUE;
    }


    /**
     *  Sets the HV bias voltage
     *
     *  @param  value  The voltage to set
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Set the HV bias voltage")
    public void setHvBias(@Argument(description="Voltage value") double value) throws PowerException
    {
        hvBiasControl.setVoltage(-Math.abs(value));
        try {
            if (powerOn[ATSChannels.CHAN_HV_BIAS] == Boolean.TRUE) {
                hvBiasControl.writeVoltage();
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns on the OTM power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the OTM")
    public void otmOn() throws PowerException
    {
        try {
            otmGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the OTM power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the OTM")
    public void otmOff() throws PowerException
    {
        try {
            otmGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns on the fan power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the fan")
    public void fanOn() throws PowerException
    {
        try {
            fanGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the fan power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the fan")
    public void fanOff() throws PowerException
    {
        try {
            fanGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns on the auxiliary power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn on the aux power")
    public void auxOn() throws PowerException
    {
        try {
            auxGroup.powerOn();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Turns off the auxiliary power.
     *
     *  @throws  PowerException
     */
    @Command(type=CommandType.ACTION, description="Turn off the aux power")
    public void auxOff() throws PowerException
    {
        try {
            auxGroup.powerOff();
        }
        finally {
            updatePowerState();
        }
    }


    /**
     *  Sets whether the fan is controlled by the temperature
     * 
     *  @param on  Whether or not temperature control is in effect
     */
    @Command(type=CommandType.ACTION, description="Set the fan temperature controlled state")
    public void setTempFanControl(boolean on)
    {
        tempFanControl = on;
        publishState();
    }


    @Command(type=CommandType.ACTION, description="Set the fan temperature limit")
    public void setTempLimit(double limit)
    {
        tempLimit = limit;
        publishState();
    }


    /**
     *  Gets the full power state.
     *
     *  @return  The full power state
     */
    @Command(type=CommandType.QUERY, description="Get the full state", level=0)
    public ATSPowerState getFullState()
    {
        return new ATSPowerState(monitorControl.getFastPeriod(), powerOn, dphiControl.getVoltage(), hvBiasControl.getVoltage(),
                                 tempChannel, tempFanControl, tempLimit, tempValue);
    }


    /**
     *  Publishes the state of the power device.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        KeyValueData kvd = new KeyValueData(ATSPowerState.KEY, getFullState());
        publishSubsystemDataOnStatusBus(kvd);
    }    


    /**
     *  Updates the power channel states.
     *
     *  Called periodically on a timer thread
     */
    private synchronized void updatePowerState()
    {
        boolean changed = monitorControl.hasPeriodChanged();
        for (int chan = 0; chan < ATSChannels.NUM_CHANNELS; chan++) {
            Boolean state = pwrControls.get(pwrChannels[chan]).readOutput();
            if (state != powerOn[chan]) {
                powerOn[chan] = state;
                changed = true;
            }
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Determines whether the non-HV REB power is fully on.
     */
    private boolean isPowerOn()
    {
        boolean isOn = true;
        for (int chan : rebChannels) {
            if (powerOn[chan] != Boolean.TRUE) {
                isOn = false;
                break;
            }
        }
        return isOn;
    }


    /**
     *  Periodically checks the controlling temperature and operates the fan.
     */
    private void controlFan()
    {
        Double temp = (Double)agentAggregatorService.getLast(tempChannel);
        if (temp == null || temp == Double.NaN) return;
        boolean changed = temp != tempValue;
        tempValue = temp;
        if (tempFanControl) {
            if (temp > tempLimit) {
                if (powerOn[ATSChannels.CHAN_FAN] == Boolean.FALSE) {
                    try {
                        fanOn();
                        changed = false;
                    }
                    catch (PowerException e) {
                        // Ignore any problem here
                    }
                }
            }
            else if (temp <= tempLimit - tempDeadZone) {
                if (powerOn[ATSChannels.CHAN_FAN] == Boolean.TRUE) {
                    try {
                        fanOff();
                        changed = false;
                    }
                    catch (PowerException e) {
                        // Ignore any problem here
                    }
                }
            }
        }
        if (changed) {
            publishState();
        }
    }

}
