package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.LinkedHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.monitor.Channel;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.subsystem.power.states.HvControlState;
import org.lsst.ccs.subsystem.power.states.RebHvControlState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.data.RebPsState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.bus.data.Alert;
import java.time.format.DateTimeFormatter;
import org.lsst.ccs.bus.states.AlertState;

// needed for data publication trigger
// public List<MonitorUpdateTask> getMonitorUpdateTasksForDevice(Device dev)
// public void forceDataPublicationOnNextUpdates(int nUpdates)

/**
 *  Implements the REB PS HV regulation system
 *
 *  @author The LSST CCS Team
 */
public class RebPsHVRegulator implements HasLifecycle {

    @LookupPath
    String HvControllerPath;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    
    private static final Logger LOG = Logger.getLogger(RebPsHVRegulator.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, RebPowerSupplyNode> rebPowers = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE,pathFilter = "(R\\d\\d)/(Reb\\d|Reb[WG])/hvbias/(DAC|[VI]befSwch)" )
    private static final Map<String,Channel> rebPowerHvChannelsMap = new HashMap<>();

    @ConfigurationParameter(name="minUpdateDelay", category="HVRegulation", units = "ms", description = "minimum time required before next update")
    protected volatile int minUpdateDelay = 20000;   // settling time after a dac change

    @ConfigurationParameter(name="staleMillis", category="HVRegulation", units = "ms", description = "max time for data freshness in V/step calc")
    protected volatile int staleMillis = 120000;   // how long is okay after skipping

    @ConfigurationParameter(name="deltaI", category="HVRegulation", units = "unitless", description = "allowed fractional excess current")
    protected volatile double deltaI = 0.02;   // allowed fractional excess current permitting changes in DAC

    @ConfigurationParameter(name="maxDeltaVolt", category="HVRegulation", units = "Volt", description = "maximum voltage step per update")
    protected volatile double maxDeltaVolt = 2.50;   // max intended jump per adjustment

    int maxStep0 = 70;
    double minSetPoint = 20.0;

    private static final Map<String,RebHvBiasInfo> hvBiasInfoMap = new HashMap<>();

    private long loopDelay = 50;
    private double voltsMax = 52.1;
    private double voltsMin = 20.0;  // minimum for controlling
    private final int dacMaxAllowed = 3200;
    private double currentLimit = 0.115;

    private long updateHvMillis = 10000;  // overridden by timers config var

    //Lock used to enforce thread visibility of rebInfo values
    private final Object rebInfoLock = new Object();

    private class RebHvBiasInfo { // inner class to hold the control params per reb

        private final Object target;
        private final Channel chVbefSwch;
        private final Channel chIbefSwch;
        private final Channel chDac;
        private final Object updateLock = new Object();

        private double setpt = 0.0;
        private double volts = 0.0;
        private double lastVolts = 0.0;
        private double current = 0.0;
        private double lastCurrent = 0.0;
        private int dac = 0;
        private int lastDac = 0;  // !=dac during time from a change until accounting is done
        private long lastMillis = 0;
        private int firstDac = 1200;
        private int maxDac = 3200;
        private double voltsPerStep = 0.100;
        private double voltsPerStepCal = 0.050;
        private double reff = 645.0;
        private boolean useVoltsPerStepCal = false;

        RebHvBiasInfo(Object target, Channel chVbefSwch, Channel chIbefSwch, Channel chDac) {
            this.target =  target;
            this.chVbefSwch = chVbefSwch;
            this.chIbefSwch = chIbefSwch;
            this.chDac = chDac;
        }
        
        public Object getUpdateLock() {
            return updateLock;
        }
        
        @Override
        public String toString() {
            synchronized( getUpdateLock() ) {
                String str = new StringBuilder()
                        .append(String.format("       setpt: %.3f\n", setpt))
                        .append(String.format("       volts: %.3f\n", volts))
                        .append(String.format("       lastVolts: %.3f\n", lastVolts))
                        .append(String.format("       current: %.3f\n", current))
                        .append(String.format("       lastCurrent: %.3f\n", lastCurrent))
                        .append(String.format("       dac: %d\n", dac))
                        .append(String.format("       lastDac: %d\n", lastDac))
                        .append(String.format("       lastMillis: %d\n", lastMillis))
                        .append(String.format("       firstDac: %d\n", firstDac))
                        .append(String.format("       maxDac: %d\n", maxDac))
                        .append(String.format("       voltsPerStep: %.3f\n", voltsPerStep))
                        .append(String.format("       voltsPerStepCal: %.3f\n", voltsPerStepCal)).toString();
                return (str);
            }
        }
    }

    List<String> allrebs = new ArrayList<>(hvBiasInfoMap.keySet());
    

    // functions
    //

    @Command(description = "Print rebInfo values for a REB", type = Command.CommandType.QUERY)
    public String getRebInfo(@Argument(description = "") String rebPath) {
        return String.format("REB: %s\n %s", rebPath, hvBiasInfoMap.get(rebPath));
    }

    public RebPsHVRegulator() {
    }

    /**
     *  Set up the RebHvBiasInfo class fixed elements
     */
    void setupRebStates()
    {
        for (Entry<String, RebPowerSupplyNode> rebNode : rebPowers.entrySet()) {

            String rebPath = rebNode.getKey();
            LOG.log(Level.INFO, String.format("rebPath: %s",rebPath));

            allrebs.add(rebPath);
            
            // construct the RebHvBiasInfo class for this reb
            hvBiasInfoMap.put(rebPath, 
                    new RebHvBiasInfo(rebNode.getValue(),
                        rebPowerHvChannelsMap.get(rebPath+"/hvbias/VbefSwch"),
                        rebPowerHvChannelsMap.get(rebPath+"/hvbias/IbefSwch"),
                        rebPowerHvChannelsMap.get(rebPath+"/hvbias/DAC")));
        }
        Collections.shuffle(allrebs);
    } 


    public void build()
    {
        LOG.log(Level.INFO, String.format("The length of rebPowerHvChannelsMap is %d",rebPowerHvChannelsMap.size()));

        // instantiate the rebInfo classes
        setupRebStates();

        //Create and schedule an AgentPeriodicTask to update the RebHVRegulationState
        AgentPeriodicTask hvControlTask = new AgentPeriodicTask("reb-hv-update",
                                   () -> updateRebHvSettings()).withPeriod(Duration.ofMillis(updateHvMillis));
        periodicTaskService.scheduleAgentPeriodicTask(hvControlTask);
    }


    /**
     *  Initialize reb hvbias regulator components
     */
    public void init()
    {
        LOG.log(Level.INFO, String.format("HV Bias Control initalizing for  %d REBs", allrebs.size()));
        //Register the state
        agentStateService.registerState(HvControlState.class, "Hv Control state", this);
        //Set the initial state
        enableHvControl(false);
    }

    protected void enableHvControl(boolean on) {
        agentStateService.updateAgentComponentState(this, on ?  HvControlState.ACTIVE : HvControlState.IDLE);
    }

    protected boolean isHvControlActive() {
        return agentStateService.isComponentInState(HvControllerPath, HvControlState.ACTIVE);
    }

    /**
     *  Updates the REB power state periodically.                                                   
     */
    public synchronized void updateRebHvSettings() {

        long t0 = System.currentTimeMillis();
        int changes = 0;
        long hvControlPeriod = periodicTaskService.getPeriodicTaskPeriod("reb-hv-update").toMillis();

        if (!isHvControlActive()) {  // global flag
            return;
        }

        // each reb adjustment takes ~10ms
        LOG.log(Level.FINE, () -> String.format("hvControlPeriod = %d", hvControlPeriod));
        loopDelay = Math.max(40, (long)((hvControlPeriod - 1000) / allrebs.size()));  // spread evenly

        for (String reb : allrebs) {

            LOG.log(Level.FINEST, () -> String.format("Updating %s",reb));
            // set up
            RebHvBiasInfo rebInfo = hvBiasInfoMap.get(reb); // rebInfo is instance of inner class
            RebPowerSupplyNode rebNode = (RebPowerSupplyNode) rebInfo.target;

            if ( agentStateService.isComponentInState(reb, RebPowerState.ON) && 
                    agentStateService.isComponentInState(reb, RebHvControlState.ALLOWED) &&
                    (System.currentTimeMillis() - rebInfo.lastMillis) > minUpdateDelay) {  // dT to settle

                // refresh config params from the RebPowerSupplyNode
                //
                synchronized (rebInfo.getUpdateLock()) {
                    rebInfo.setpt = rebNode.getHvBiasSetPoint();
                    rebInfo.maxDac = rebNode.getMaxDac();
                    rebInfo.firstDac = rebNode.getFirstDac();
                    rebInfo.reff = rebNode.getReff();
                    rebInfo.voltsPerStepCal = rebNode.getVoltsPerStepCal();   // calibrated value
                    rebInfo.useVoltsPerStepCal = rebNode.getUseVoltsPerStepCal();

                    // Update values from trending
                    //
                    rebInfo.dac = (int) rebInfo.chDac.getValue(); // no warn needed
                    if (rebInfo.dac == 0) {  // reset
                        rebInfo.voltsPerStep = 0.100;
                    }

                    rebInfo.volts = rebInfo.chVbefSwch.getValue();;
                    if (rebInfo.volts < 0.0 || rebInfo.volts > voltsMax) {
                        LOG.log(Level.SEVERE, () -> String.format("%s hvbias getValue():%s is out of allowed range: 0--%f",
                                reb, rebInfo.volts, voltsMax));
                    }

                    rebInfo.current = rebInfo.chIbefSwch.getValue();
                    if (rebInfo.current < 0.0 || rebInfo.current > currentLimit) {
                        LOG.log(Level.SEVERE, () -> String.format("%s hvbias current:%s out of allowed range: 0--%f",
                                reb, rebInfo.current, currentLimit));
                    }
                    LOG.log(Level.FINE, () -> String.format("%s latest values: dac: %d  volts: %6.3f current: %.3f (mA)",
                            reb, rebInfo.dac, rebInfo.volts, rebInfo.current));

                    // Determine new running value for voltsPerStep if criteria pass
                    //
                    double deltaDac = rebInfo.dac - rebInfo.lastDac;  // non-zero ==> update
                    if (deltaDac != 0 && System.currentTimeMillis() - rebInfo.lastMillis < staleMillis) {

                        double deltaVolts = rebInfo.volts - rebInfo.lastVolts;  // now - lastMillis
                        double deltaCurrent = rebInfo.current - rebInfo.lastCurrent;  // now - lastMillis
                        double oldVal = rebInfo.voltsPerStep;
                        double newVal = deltaVolts / deltaDac;

                        if (newVal > 0.025) { // minimum useful, may need max also
                            rebInfo.voltsPerStep = (2.0 * oldVal + newVal) / 3.0; // fold-in
                            rebInfo.lastDac = rebInfo.dac;  // ==> next deltaDac is zero
                            LOG.log(Level.INFO, () -> String.format("%s: Update running voltsPerStep: (%.3f,%.3f) --> %.3f",
                                    reb, oldVal, newVal, rebInfo.voltsPerStep));
                        } else {
                            if (deltaVolts < -0.1 && deltaCurrent > 0.005) { // may indicate overlight condition
                                LOG.log(Level.WARNING, () -> String.format("%s V/step: %.3f, dI: %.3f, dV: %.3f overlight condition?",
                                        reb, newVal, deltaCurrent, deltaVolts));
                            }
                        }
                    } else {
                        rebInfo.lastDac = rebInfo.dac;  // stale ==> next deltaDac is zero
                    }

                    // Determine if a change in DAC value should be made
                    //
                    int newDac = 0;
                    if (rebInfo.setpt > minSetPoint) {

                        int steps = 0;
                        int maxStep = (int) (maxStep0 - 1.1 * rebInfo.volts);  // 70@0V to 15@50V
                        if (maxStep < 0) {
                            maxStep = 0;
                        }
                        maxStep = Math.min(maxStep, (int) (maxDeltaVolt / rebInfo.voltsPerStep));

                        // get provisional value followed by sequence of filters for safety
                        if (!rebInfo.useVoltsPerStepCal || (Math.abs(rebInfo.setpt - rebInfo.volts) > 2.0)) {
                            steps = Math.min(maxStep, (int) Math.round(((rebInfo.setpt - rebInfo.volts) / rebInfo.voltsPerStep)));
                        } else {
                            steps = Math.min(maxStep, (int) Math.round(((rebInfo.setpt - rebInfo.volts) / rebInfo.voltsPerStepCal)));
                        }
                        if (steps != 0) {
                            LOG.log(Level.FINE, String.format("%s: using %d DAC steps of %d maximum", reb, steps, maxStep));
                        }

                        newDac = rebInfo.dac + steps;  // provisional value, apply limits
                        newDac = Math.max(newDac, 0);  // >=0
                        newDac = Math.min(newDac, rebInfo.maxDac);  // <~53V

                        // first step is defined for ~5V
                        if (newDac > 0 && newDac < rebInfo.firstDac) {
                            newDac = rebInfo.firstDac;
                            LOG.log(Level.INFO, () -> String.format("%s: start ramp up from Dac: %d", reb, rebInfo.firstDac));
                        } else {
                            LOG.log(Level.FINE, String.format("%s: using %d DAC steps of %d maximum",
                                    reb, newDac - rebInfo.dac, maxStep));
                        }
                    } else {
                        if (rebInfo.setpt > 0.0) {
                            LOG.log(Level.WARNING, () -> String.format(
                                    "%s: setpt(%.3f) < minSetPoint(%.3f), using 0", reb, rebInfo.setpt, minSetPoint));
                        }
                        newDac = 0;  // anything below minSetPoint is squashed to zero
                    }

                    // Make the change and update rebInfo for the next round
                    //
                    if (newDac != rebInfo.dac) {  // ==> change

                        // veto positive change if current is higher than expected ==> spike or overlight
                        double currentMax = (1.0 + deltaI) * (rebInfo.setpt + rebInfo.volts) / 2.0 / rebInfo.reff;
                        if (newDac > rebInfo.dac && rebInfo.current > currentMax) {
                            LOG.log(Level.WARNING,
                                    () -> String.format("Skipping update on %s, current: %.3f > %.3f mA limit",
                                            reb, rebInfo.current, currentMax));
                        } else {
                            try {
                                LOG.log(Level.FINE, String.format("%s hvBias DAC changing from %d to %d", reb, rebInfo.dac, newDac));
                                rebNode.setHvBiasDac(newDac, true);   // throws
                                changes += 1;

                                // updates in rebInfo to support next iteration, trigger voltsPerStep update
                                if (newDac > 0) {
                                    rebInfo.lastDac = rebInfo.dac;
                                    rebInfo.dac = newDac;
                                    rebInfo.lastVolts = rebInfo.volts;
                                    rebInfo.lastCurrent = rebInfo.current;
                                } else {
                                    rebInfo.lastDac = 0;
                                    rebInfo.dac = 0;
                                    rebInfo.lastVolts = 0.0;
                                    rebInfo.lastCurrent = 0.0;
                                }
                                rebInfo.lastMillis = System.currentTimeMillis();
                            } catch (PowerException ex) {
                                LOG.log(Level.SEVERE, "Failed to set HV bias dac: " + ex);
                            }
                        }
                    }
                }                
            } else {
                LOG.log(Level.FINEST, () -> String.format("Skipping %s: OFF|BLOCKED|tooEarly", reb));
            }

            try {
                Thread.sleep((long)loopDelay); // delay between rebs
            } catch (InterruptedException ex) {
                LOG.log(Level.SEVERE, "Sleep interrupted");
            }
        }

        long t1 = System.currentTimeMillis();
        long deltaTime = t1 - t0;
        if (changes > 0) {
            LOG.log(Level.FINE, String.format( "loopTime=%f changeCount=%d loopDelay=%d", deltaTime/1000., changes, loopDelay));
        }
    }
}
